/*
 * Decompiled with CFR 0.152.
 */
package org.zowe.apiml.caching.service.vsam;

import java.io.Closeable;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.regex.Pattern;
import lombok.Generated;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.zowe.apiml.caching.service.vsam.RetryableVsamException;
import org.zowe.apiml.caching.service.vsam.VsamInitializer;
import org.zowe.apiml.caching.service.vsam.VsamKey;
import org.zowe.apiml.caching.service.vsam.VsamRecord;
import org.zowe.apiml.caching.service.vsam.VsamRecordException;
import org.zowe.apiml.caching.service.vsam.ZFileProducer;
import org.zowe.apiml.caching.service.vsam.config.VsamConfig;
import org.zowe.apiml.message.log.ApimlLogger;
import org.zowe.apiml.zfile.ZFile;
import org.zowe.apiml.zfile.ZFileException;

public class VsamFile
implements Closeable {
    @Generated
    private static final Logger log = LoggerFactory.getLogger(VsamFile.class);
    private final ZFile zfile;
    private final VsamConfig vsamConfig;
    private final ZFileProducer zFileProducer;
    private final ApimlLogger apimlLog;
    public static final String VSAM_RECORD_ERROR_MESSAGE = "VsamRecordException occurred: {}";
    public static final String RECORD_FOUND_MESSAGE = "Record found: {}";
    public static final String RECORD_CANNOT_BE_NULL_MESSAGE = "Record cannot be null";
    public static final String UNSUPPORTED_ENCODING_MESSAGE = "Unsupported encoding: {}";
    private static final String ERROR_INITIALIZING_STORAGE_MESSAGE_KEY = "org.zowe.apiml.cache.errorInitializingStorage";
    private static final String STORAGE_TYPE = "VSAM";
    private static final Pattern REGEX_CORRECT_FILENAME = Pattern.compile("^//'.*'");

    public VsamFile(VsamConfig config, VsamConfig.VsamOptions options, ApimlLogger apimlLogger) {
        this(config, options, false, apimlLogger);
    }

    public VsamFile(VsamConfig config, VsamConfig.VsamOptions options, boolean initialCreation, ApimlLogger apimlLogger) {
        this(config, options, initialCreation, new ZFileProducer(config, options, apimlLogger), new VsamInitializer(), apimlLogger);
    }

    public VsamFile(VsamConfig config, VsamConfig.VsamOptions options, boolean initialCreation, ZFileProducer zFileProducer, VsamInitializer vsamInitializer, ApimlLogger apimlLogger) {
        this.apimlLog = apimlLogger;
        if (config == null) {
            this.apimlLog.log(ERROR_INITIALIZING_STORAGE_MESSAGE_KEY, new Object[]{"vsam", "wrong Configuration", "No configuration provided"});
            throw new IllegalArgumentException("Cannot create VsamFile with null configuration");
        }
        this.vsamConfig = config;
        log.info("VsamFile::new with parameters: {}, Vsam options: {}", (Object)this.vsamConfig, (Object)options);
        if (!REGEX_CORRECT_FILENAME.matcher(this.vsamConfig.getFileName()).find()) {
            String nonConformance = "VsamFile name does not conform to //'VSAM.DATASET.NAME' pattern  " + this.vsamConfig.getFileName();
            this.apimlLog.log(ERROR_INITIALIZING_STORAGE_MESSAGE_KEY, new Object[]{"vsam", "wrong vsam name: " + this.vsamConfig.getFileName(), nonConformance});
            throw new IllegalArgumentException(nonConformance);
        }
        this.zFileProducer = zFileProducer;
        try {
            this.zfile = this.openZfile();
            if (initialCreation) {
                log.info("Warming up VSAM file");
                vsamInitializer.warmUpVsamFile(this.zfile, this.vsamConfig);
            }
        }
        catch (Exception e) {
            String info = String.format("opening of %s in mode %s failed", new Object[]{this.vsamConfig, options});
            if (initialCreation) {
                this.apimlLog.log(ERROR_INITIALIZING_STORAGE_MESSAGE_KEY, new Object[]{STORAGE_TYPE, info, e});
            } else {
                log.info(info);
            }
            throw new IllegalStateException("Failed to open VsamFile");
        }
    }

    @Override
    public void close() {
        if (this.zfile != null) {
            try {
                this.zfile.close();
            }
            catch (ZFileException e) {
                this.apimlLog.log("org.zowe.apiml.cache.errorQueryingStorage", new Object[]{STORAGE_TYPE, "Closing ZFile failed: " + e});
            }
        }
    }

    public Optional<VsamRecord> create(final VsamRecord vsamRec) {
        log.info("Attempting to create record: {}", (Object)vsamRec);
        return this.recordOperation(vsamRec, new RecordHandler(){

            @Override
            public Optional<VsamRecord> handleRecordFound() {
                log.info("The record already exists and will not be created. Use update instead.");
                return Optional.empty();
            }

            @Override
            public Optional<VsamRecord> handleNoRecordFound() throws VsamRecordException, ZFileException {
                log.info("Writing Record: {}", (Object)vsamRec);
                VsamFile.this.zfile.write(vsamRec.getBytes());
                return Optional.of(vsamRec);
            }
        });
    }

    public Optional<VsamRecord> read(VsamRecord vsamRec) {
        log.info("Attempting to read record: {}", (Object)vsamRec);
        return this.recordOperation(vsamRec, () -> {
            byte[] recBuf = new byte[this.vsamConfig.getRecordLength()];
            this.zfile.read(recBuf);
            log.trace("RecBuf: {}", (Object)recBuf);
            log.info("ConvertedStringValue: {}", (Object)new String(recBuf, this.vsamConfig.getEncoding()));
            VsamRecord returned = new VsamRecord(this.vsamConfig, recBuf);
            log.info("VsamRecord read: {}", (Object)returned);
            return Optional.of(returned);
        });
    }

    public Optional<VsamRecord> update(VsamRecord vsamRec) {
        log.info("Attempting to update record: {}", (Object)vsamRec);
        return this.recordOperation(vsamRec, () -> {
            byte[] recBuf = new byte[this.vsamConfig.getRecordLength()];
            this.zfile.read(recBuf);
            log.trace("Read found record: {}", (Object)new String(recBuf, "IBM-1047"));
            log.info("Will update record: {}", (Object)vsamRec);
            int nUpdated = this.zfile.update(vsamRec.getBytes());
            log.info("ZFile.update return value: {}", (Object)nUpdated);
            return Optional.of(vsamRec);
        });
    }

    public Optional<VsamRecord> delete(VsamRecord vsamRec) {
        log.info("Attempting to delete record: {}", (Object)vsamRec);
        return this.recordOperation(vsamRec, () -> {
            byte[] recBuf = new byte[this.vsamConfig.getRecordLength()];
            this.zfile.read(recBuf);
            VsamRecord returned = new VsamRecord(this.vsamConfig, recBuf);
            this.zfile.delrec();
            log.info("Deleted vsam record: {}", (Object)returned);
            return Optional.of(returned);
        });
    }

    private Optional<VsamRecord> recordOperation(VsamRecord vsamRec, RecordHandler recordHandler) {
        if (vsamRec == null) {
            throw new IllegalArgumentException(RECORD_CANNOT_BE_NULL_MESSAGE);
        }
        try {
            boolean found = this.zfile.locate(vsamRec.getKeyBytes(), 3);
            if (found) {
                return recordHandler.handleRecordFound();
            }
            return recordHandler.handleNoRecordFound();
        }
        catch (UnsupportedEncodingException e) {
            log.info(UNSUPPORTED_ENCODING_MESSAGE, (Object)"IBM-1047");
        }
        catch (ZFileException e) {
            log.info(e.toString());
            throw new RetryableVsamException(e);
        }
        catch (VsamRecordException e) {
            log.info(VSAM_RECORD_ERROR_MESSAGE, (Object)e.toString());
            throw new RetryableVsamException(e);
        }
        return Optional.empty();
    }

    public List<VsamRecord> readForService(String serviceId) {
        ArrayList<VsamRecord> returned = new ArrayList<VsamRecord>();
        this.serviceWideOperation(serviceId, (zfile, vsamRec) -> {
            log.debug("Retrieve the record");
            returned.add(vsamRec);
        });
        return returned;
    }

    public void deleteForService(String serviceId) {
        this.serviceWideOperation(serviceId, (zfile, vsamRec) -> {
            log.debug("Delete the record");
            zfile.delrec();
        });
    }

    private void serviceWideOperation(String serviceId, ServiceWideOperation operation) {
        if (serviceId == null || serviceId.isEmpty()) {
            throw new IllegalArgumentException("serviceId cannot be null");
        }
        VsamKey key = new VsamKey(this.vsamConfig);
        try {
            byte[] recBuf = new byte[this.vsamConfig.getRecordLength()];
            String keyGe = key.getKeySidOnly(serviceId);
            log.info("Attempt to find key in KEY_GE mode: {}", (Object)keyGe);
            boolean found = this.zfile.locate(key.getKeyBytesSidOnly(serviceId), 5);
            log.info(RECORD_FOUND_MESSAGE, (Object)found);
            int overflowProtection = 10000;
            while (found) {
                int nread = this.zfile.read(recBuf);
                log.trace("RecBuf: {}", (Object)recBuf);
                log.info("nread: {}", (Object)nread);
                String convertedStringValue = new String(recBuf, "IBM-1047");
                VsamRecord vsamRec = new VsamRecord(this.vsamConfig, recBuf);
                log.info("Read record: {}", (Object)vsamRec);
                if (nread < 0) {
                    log.info("nread is < 0, stopping the retrieval");
                    found = false;
                    continue;
                }
                log.trace("convertedStringValue: >{}<", (Object)convertedStringValue);
                log.trace("keyGe: >{}<", (Object)keyGe);
                if (!convertedStringValue.trim().startsWith(keyGe.trim())) {
                    log.info("read record does not start with serviceId's keyGe, stopping the retrieval");
                    found = false;
                    continue;
                }
                log.info("read record starts with serviceId's keyGe, retrieving this record");
                operation.resolveValidRecord(this.zfile, vsamRec);
                if (--overflowProtection > 0) continue;
                log.info("Maximum number of records retrieved, stopping the retrieval");
                break;
            }
        }
        catch (UnsupportedEncodingException e) {
            log.info(UNSUPPORTED_ENCODING_MESSAGE, (Object)"IBM-1047");
        }
        catch (ZFileException e) {
            log.info(e.toString());
        }
        catch (VsamRecordException e) {
            log.info(VSAM_RECORD_ERROR_MESSAGE, (Object)e.toString());
        }
    }

    public Optional<byte[]> readBytes(byte[] arrayToStoreIn) throws ZFileException {
        if (this.getZfile().read(arrayToStoreIn) == -1) {
            return Optional.empty();
        }
        return Optional.of(arrayToStoreIn);
    }

    public Integer countAllRecords() {
        int recordsCounter = 0;
        try {
            byte[] recBuf = new byte[this.vsamConfig.getRecordLength()];
            int overflowProtection = 10000;
            while (this.zfile.read(recBuf) != -1) {
                log.trace("RecBuf: {}", (Object)recBuf);
                ++recordsCounter;
                if (--overflowProtection > 0) continue;
                log.info("Maximum number of records retrieved, stopping the retrieval");
                break;
            }
        }
        catch (ZFileException e) {
            log.info(e.toString());
        }
        return recordsCounter;
    }

    private ZFile openZfile() throws VsamRecordException {
        return this.zFileProducer.openZfile();
    }

    @Generated
    public ZFile getZfile() {
        return this.zfile;
    }

    @FunctionalInterface
    private static interface RecordHandler {
        public Optional<VsamRecord> handleRecordFound() throws VsamRecordException, ZFileException, UnsupportedEncodingException;

        default public Optional<VsamRecord> handleNoRecordFound() throws VsamRecordException, ZFileException {
            log.info("No record found");
            return Optional.empty();
        }
    }

    @FunctionalInterface
    private static interface ServiceWideOperation {
        public void resolveValidRecord(ZFile var1, VsamRecord var2) throws ZFileException;
    }
}

