package website.dachuan.migration.spring;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.Order;
import website.dachuan.migration.MigrationRun;
import website.dachuan.migration.MigrationTenantDataRun;
import website.dachuan.migration.entity.TenantEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ITenantService;
import website.dachuan.migration.utils.Files;

import javax.annotation.PreDestroy;
import javax.sql.DataSource;
import java.io.Closeable;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * @author yqb22
 */
@Order(1)
@Slf4j
public class MigrationRunner implements ApplicationRunner, Closeable {
    private final MigrationProps props;
    private final ITenantService tenantService;
    private final MigrationRun migrationRun;
    private final MigrationTenantDataRun migrationTenantDataRunRun;
    private final Map<String, DataSource> dataSourceMap = new HashMap<>();

    public MigrationRunner(ApplicationContext applicationContext, MigrationProps props, ITenantService tenantService,
                           MigrationRun migrationRun, MigrationTenantDataRun migrationTenantDataRunRun) {
        this.props = props;
        this.tenantService = tenantService;
        this.migrationRun = migrationRun;
        this.migrationTenantDataRunRun = migrationTenantDataRunRun;
        Map<String, DataSource> dssMap = applicationContext.getBeansOfType(DataSource.class);
        if (dssMap.isEmpty()) {
            return;
        }
        try {
            Class.forName("com.baomidou.dynamic.datasource.DynamicRoutingDataSource", false, getClass().getClassLoader());
            for (Map.Entry<String, DataSource> entry : dssMap.entrySet()) {
                if (entry.getValue() instanceof com.baomidou.dynamic.datasource.DynamicRoutingDataSource) {
                    this.dataSourceMap.putAll(((com.baomidou.dynamic.datasource.DynamicRoutingDataSource) entry.getValue()).getDataSources());
                } else if (entry.getValue() instanceof website.dachuan.migration.DynamicRoutingDataSource) {
                    this.dataSourceMap.putAll(((website.dachuan.migration.DynamicRoutingDataSource) entry.getValue()).getDataSources());
                } else {
                    dataSourceMap.put(entry.getKey(), entry.getValue());
                }
            }
        } catch (ClassNotFoundException e) {
            log.debug("com.baomidou.dynamic.datasource.DynamicRoutingDataSource ClassNotFoundException");
            for (Map.Entry<String, DataSource> entry : dssMap.entrySet()) {
                if (entry.getValue() instanceof website.dachuan.migration.DynamicRoutingDataSource) {
                    this.dataSourceMap.putAll(((website.dachuan.migration.DynamicRoutingDataSource) entry.getValue()).getDataSources());
                } else {
                    dataSourceMap.put(entry.getKey(), entry.getValue());
                }
            }
        }
    }

    /**
     * 执行数据迁移
     *
     * @param args 参数
     */
    @Override
    public void run(ApplicationArguments args) throws Exception {
        // 清理 migration临时文件
        Files.cleanTemp();
        if (dataSourceMap.keySet().size() == 0) {
            return;
        }
        // 复制sql脚本到临时文件夹中 文件逻辑，如果在文件系统中可以获取到就是文件系统中文件如果文件系统中获取不到就是 class path 下相对路径文件夹
        // 解决在jar包下面可能存在各自升级sql的问题,需要查找所有地址进行获取
        Files.copyFile(this.props.getScriptDirPath());
        // 复制租户数据文件
        Files.copyFile(this.props.getTenantData().getScriptDirPath());
        // 执行ddl 语句
        for (Map.Entry<String, DataSource> e : dataSourceMap.entrySet()) {
            try {
                this.migrationRun.run(e.getValue().getConnection());
            } catch (Exception exception) {
                log.error("migrationRun error", exception);
                System.exit(0);
            }
        }
        if (!props.getTenantData().isEnabled()) {
            dataSourceMap.clear();
            return;
        }
        // 多租户数据执行
        String primaryDataBaseId = props.getTenantData().getPrimaryDataBaseId();
        String tenantTable = props.getTenantData().getTenantTableName();
        String tenantTableId = props.getTenantData().getTenantTableIdColumn();
        String tenantJdbcUrl = props.getTenantData().getJdbcUrlColumn();
        DataSource primary = dataSourceMap.get(primaryDataBaseId);
        if (primary == null) {
            throw new RuntimeException("多租户模式下，未查询到主数据源，请检查配置项是否正确！");
        }
        if (primary.getConnection() == null) {
            throw new RuntimeException("多租户模式下，主数据源连接异常，请检查配置项是否正确！");
        }
        // 执行超级租户数据
        this.migrationTenantDataRunRun.run(primary.getConnection(), null, true);
        // 执行各个租户下数据
        List<TenantEntity> tenantEntities = tenantService.queryTenantInfo(primary.getConnection(), tenantTable, tenantTableId, tenantJdbcUrl);
        if (tenantEntities == null || tenantEntities.size() < 1) {
            dataSourceMap.clear();
            return;
        }
        for (Map.Entry<String, DataSource> e : dataSourceMap.entrySet()) {
            List<String> tenantIds;
            if (props.getTenantData().getPrimaryDataBaseId().equals(e.getKey())) {
                tenantIds = tenantEntities.stream()
                        .filter(t -> !t.isIndependentDatabase())
                        .filter(t -> t.getTenantId() != null && t.getTenantId().length() > 0)
                        .map(TenantEntity::getTenantId).collect(Collectors.toList());
            } else {
                tenantIds = tenantEntities.stream()
                        .filter(TenantEntity::isIndependentDatabase)
                        .filter(t -> t.getTenantId() != null && t.getTenantId().length() > 0)
                        .filter(t -> {
                                    try {
                                        return t.getUrl().equals(e.getValue().getConnection().getMetaData().getURL());
                                    } catch (SQLException sqlException) {
                                        log.error(log.getName(), sqlException);
                                        return false;
                                    }
                                }
                        ).map(TenantEntity::getTenantId).collect(Collectors.toList());
            }
            this.migrationTenantDataRunRun.run(e.getValue().getConnection(), tenantIds, false);
        }
        dataSourceMap.clear();
    }

    @Override
    public void close() {
        dataSourceMap.clear();
    }

    @PreDestroy
    public void destroy() {
        boolean t = Files.deleteTemp();
        if (!t) {
            log.error("migration 文件夹删除失败！");
        }
    }
}
