package website.dachuan.migration.spring;

import com.baomidou.dynamic.datasource.DynamicRoutingDataSource;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;
import website.dachuan.migration.MigrationRun;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISqlScriptRunner;

import javax.sql.DataSource;
import java.io.Closeable;
import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

@Order(1)
@Slf4j
public class MigrationRunner implements ApplicationRunner, Closeable {
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;
    private final MigrationRun migrationRun;
    private final Map<String, DataSource> dataSourceMap = new HashMap<>();

    public MigrationRunner(ApplicationContext applicationContext, MigrationProps props, ISqlScriptRunner scriptRunner,
                           MigrationRun migrationRun) {
        this.props = props;
        this.scriptRunner = scriptRunner;
        this.migrationRun = migrationRun;
        Map<String, DataSource> dssMap = applicationContext.getBeansOfType(DataSource.class);
        if (dssMap.isEmpty()) {
            return;
        }
        boolean cbddd = false;
        try {
            Class.forName("com.baomidou.dynamic.datasource.DynamicRoutingDataSource", false, getClass().getClassLoader());
            cbddd = true;
        } catch (ClassNotFoundException e) {
            log.debug("com.baomidou.dynamic.datasource.DynamicRoutingDataSource ClassNotFoundException");
        }
        for (Map.Entry<String, DataSource> entry : dssMap.entrySet()) {
            if (cbddd && entry.getValue() instanceof DynamicRoutingDataSource) {
                this.dataSourceMap.putAll(((DynamicRoutingDataSource) entry.getValue()).getDataSources());
            } else if (entry.getValue() instanceof website.dachuan.migration.DynamicRoutingDataSource) {
                this.dataSourceMap.putAll(((website.dachuan.migration.DynamicRoutingDataSource) entry.getValue()).getDataSources());
            } else {
                dataSourceMap.put(entry.getKey(), entry.getValue());
            }
        }
    }

    /**
     * 执行数据迁移
     *
     * @param args 参数
     */
    @Override
    public void run(ApplicationArguments args) {
        if (dataSourceMap.keySet().size() == 0) {
            return;
        }
        // 多租户数据执行
        List<Map<String, Object>> r = null;
        if (props.getTenantData().isEnabled()) {
            String primaryDataBaseId = props.getTenantData().getPrimaryDataBaseId();
            String tenantTable = props.getTenantData().getTenantTableName();
            String tenantTableId = props.getTenantData().getTenantTableIdColumn();
            String tenantJdbcUrl = props.getTenantData().getJdbcUrlColumn();
            StringBuilder sql = new StringBuilder("select ")
                    .append(tenantTableId).append(" as tenantId, ")
                    .append(tenantJdbcUrl).append(" as url ")
                    .append(" from ").append(tenantTable);
            DataSource primary = dataSourceMap.get(primaryDataBaseId);
            if (primary == null) {
                throw new IllegalArgumentException("主数据源获取失败！");
            }
            try {
                r = scriptRunner.runScript(primary.getConnection(), sql.toString());
            } catch (Exception e) {
                log.error("从主数据源中获取租户信息异常：", e);
            }
        }
        for (Map.Entry<String, DataSource> e : dataSourceMap.entrySet()) {
            List<String> tenantIds = new ArrayList<>();
            if (props.getTenantData().isEnabled() && r != null) {
                if (props.getTenantData().getPrimaryDataBaseId().equals(e.getKey())) {
                    tenantIds = r.stream()
                            .filter(stringObjectMap -> StringUtils.isEmpty(stringObjectMap.get("url")))
                            .map(stringObjectMap -> {
                                Object tenantId = stringObjectMap.get("tenantId");
                                if (tenantId != null) {
                                    return tenantId.toString();
                                } else {
                                    return null;
                                }
                            }).filter(Objects::nonNull).collect(Collectors.toList());
                } else {
                    tenantIds = r.stream()
                            .filter(stringObjectMap -> {
                                try {
                                    return !StringUtils.isEmpty(stringObjectMap.get("url")) && stringObjectMap.get("url").equals(e.getValue().getConnection().getMetaData().getURL());
                                } catch (SQLException sqlException) {
                                    log.error(log.getName(), sqlException);
                                    return false;
                                }
                            })
                            .map(stringObjectMap -> {
                                Object tenantId = stringObjectMap.get("tenantId");
                                if (tenantId != null) {
                                    return tenantId.toString();
                                } else {
                                    return null;
                                }
                            }).filter(Objects::nonNull).collect(Collectors.toList());
                }
            } else {
                log.error("系统开启多租户数据但是主数据未初始化，未查询到租户信息无法进行多租户数据处理！");
            }
            this.migrationRun.run(e.getValue(), tenantIds);
        }
        dataSourceMap.clear();
    }

    @Override
    public void close() {
        dataSourceMap.clear();
    }
}
