package website.dachuan.migration.spring;

import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.annotation.Order;
import org.springframework.util.StringUtils;
import website.dachuan.migration.MigrationProcess;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.IDBOperation;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.utils.DatabaseIdProvider;

import javax.sql.DataSource;
import java.io.Closeable;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

@Order(1)
@Slf4j
public class MigrationRunner implements ApplicationRunner, Closeable {
    private final Map<String, DataSource> dataSource;
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;
    private final ISchemaHistoryService schemaHistoryService;
    private final Map<String, IDBOperation> idbOperationMap;
    private MigrationProcess process;

    public MigrationRunner(Map<String, DataSource> dataSource, MigrationProps props, ISqlScriptRunner scriptRunner,
                           ISchemaHistoryService schemaHistoryService, Map<String, IDBOperation> idbOperationMap) {
        this.dataSource = dataSource;
        this.props = props;
        this.scriptRunner = scriptRunner;
        this.schemaHistoryService = schemaHistoryService;
        this.idbOperationMap = idbOperationMap;
    }

    /**
     * 执行数据迁移
     *
     * @param args 参数
     * @throws Exception 异常
     */
    @Override
    public void run(ApplicationArguments args) throws Exception {
        if (dataSource == null || dataSource.keySet().size() == 0) {
            return;
        }
        process = new MigrationProcess(props, idbOperationMap, scriptRunner, schemaHistoryService);
        // 多租户数据执行
        List<Map<String, Object>> r = null;
        if (props.getTenantData().isEnabled()) {
            String primaryDataBaseId = props.getTenantData().getPrimaryDataBaseId();
            String tenantTable = props.getTenantData().getTenantTableName();
            String tenantTableId = props.getTenantData().getTenantTableIdColumn();
            String tenantJdbcUrl = props.getTenantData().getJdbcUrlColumn();
            StringBuilder sql = new StringBuilder("select ")
                    .append(tenantTableId).append(" as tenantId, ")
                    .append(tenantJdbcUrl).append(" as url ")
                    .append(" from ").append(tenantTable);
            DataSource primary = dataSource.get(primaryDataBaseId);
            if (primary == null) {
                throw new IllegalArgumentException("主数据源获取失败！");
            }
            try {
                r = scriptRunner.runScript(primary.getConnection(), sql.toString());
            } catch (Exception e) {
                log.error("从主数据源中获取租户信息异常：", e);
            }
        }
        for (Map.Entry<String, DataSource> e : dataSource.entrySet()) {
            List<String> tenantIds = new ArrayList<>();
            if (props.getTenantData().isEnabled() && r != null) {
                if (props.getTenantData().getPrimaryDataBaseId().equals(e.getKey())) {
                    tenantIds = r.stream()
                            .filter(stringObjectMap -> StringUtils.isEmpty(stringObjectMap.get("url")))
                            .map(stringObjectMap -> {
                                Object tenantId = stringObjectMap.get("tenantId");
                                if (tenantId != null) {
                                    return tenantId.toString();
                                } else {
                                    return null;
                                }
                            }).filter(Objects::nonNull).collect(Collectors.toList());
                } else {
                    tenantIds = r.stream()
                            .filter(stringObjectMap -> {
                                try {
                                    return !StringUtils.isEmpty(stringObjectMap.get("url")) && stringObjectMap.get("url").equals(e.getValue().getConnection().getMetaData().getURL());
                                } catch (SQLException sqlException) {
                                    log.error(log.getName(), sqlException);
                                    return false;
                                }
                            })
                            .map(stringObjectMap -> {
                                Object tenantId = stringObjectMap.get("tenantId");
                                if (tenantId != null) {
                                    return tenantId.toString();
                                } else {
                                    return null;
                                }
                            }).filter(Objects::nonNull).collect(Collectors.toList());
                }
            } else {
                log.error("系统开启多租户数据但是主数据未初始化，未查询到租户信息无法进行多租户数据处理！");
            }
            this.runMigration(e.getValue(), tenantIds);
        }
        dataSource.clear();
    }

    private void runMigration(DataSource dataSource, List<String> tenantIds) {
        Connection conn;
        DatabaseIdProvider.DatabaseId databaseId;
        try {
            conn = dataSource.getConnection();
            String databaseProductName = DatabaseIdProvider.getDatabaseId(conn);
            databaseId = DatabaseIdProvider.DatabaseId.valueOf(databaseProductName);
        } catch (SQLException e) {
            log.error("", e);
            return;
        }
        try {
            if (databaseId.ddlTransaction) {
                conn.setAutoCommit(false);
            }
            process.exec(conn, tenantIds);
            if (databaseId.ddlTransaction) {
                conn.commit();
            }
        } catch (Exception e) {
            log.error("", e);
            try {
                if (databaseId.ddlTransaction) {
                    conn.rollback();
                }
                conn.close();
            } catch (SQLException sqlException) {
                log.error("", sqlException);
            }
            this.close();
            System.exit(0);
        }
    }

    @Override
    public void close() {
        dataSource.clear();
    }
}
