package website.dachuan.migration.spring;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.MigrationProcess;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.IDBOperation;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.annotation.Order;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

@Order(1)
@Slf4j
public class MigrationRunner implements ApplicationRunner {
    private Map<String, DataSource> dataSource;
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;
    private final ISchemaHistoryService schemaHistoryService;
    private final List<String> canDdlAtomicity = new ArrayList<>(2);
    @Autowired
    private Map<String, IDBOperation> idbOperationMap;


    public MigrationRunner(Map<String, DataSource> dataSource, MigrationProps props, ISqlScriptRunner scriptRunner, ISchemaHistoryService schemaHistoryService) {
        this.dataSource = dataSource;
        this.props = props;
        this.scriptRunner = scriptRunner;
        this.schemaHistoryService = schemaHistoryService;
        this.canDdlAtomicity.add("PostgreSQL");
    }

    /**
     * 执行数据迁移
     *
     * @param args 参数
     * @throws Exception 异常
     */
    @Override
    public void run(ApplicationArguments args) throws Exception {
        if (dataSource == null || dataSource.keySet().size() == 0) {
            return;
        }
        for (Map.Entry<String, DataSource> e : dataSource.entrySet()) {
            this.runMigration(e.getValue());
        }
        dataSource.clear();
        dataSource = null;
    }

    private void runMigration(DataSource dataSource) throws Exception {
        Connection conn = null;
        boolean atomicity = false;
        try {
            conn = dataSource.getConnection();
            String databaseProductName = conn.getMetaData().getDatabaseProductName();
            IDBOperation dbOperation = idbOperationMap.get(databaseProductName);
            atomicity = canDdlAtomicity.stream().allMatch(databaseProductName::equals);
            if (atomicity) {
                conn.setAutoCommit(false);
            }
            MigrationProcess process = new MigrationProcess(props, dbOperation, scriptRunner, schemaHistoryService);
            process.exec(conn);
            if (atomicity) {
                conn.commit();
            }
        } catch (Exception e) {
            log.error("", e);
            try {
                assert conn != null;
                if (atomicity) {
                    conn.rollback();
                }
                conn.close();
            } catch (SQLException sqlException) {
                log.error("", sqlException);
            }
            throw e;
        }
    }
}
