package website.dachuan.migration.utils;

import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.bo.SuccessNum;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
public class SqlScriptCommon {
    private final ISchemaHistoryService schemaHistoryService;
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;

    public SqlScriptCommon(MigrationProps props, ISchemaHistoryService schemaHistoryService, ISqlScriptRunner scriptRunner) {
        this.schemaHistoryService = schemaHistoryService;
        this.props = props;
        this.scriptRunner = scriptRunner;
    }

    public List<SQLScriptBo> listTenantDir(Connection conn, File folder, boolean incremental, String tenantId) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts = new ArrayList<>();
        if (folder.exists()) {
            // 检索公用文件夹下文件
            File common = new File(folder.getAbsolutePath() + File.separator + "common");
            if (common.exists()) {
                sqlScripts.addAll(this.listSqlScriptBo(conn, (incremental ? "common_" : "deploy_init_common_") + tenantId, common, tenantId, true));
            }
            // 检索租户文件夹下文件
            File tenant = new File(folder.getAbsolutePath() + File.separator + tenantId);
            if (tenant.exists()) {
                sqlScripts.addAll(this.listSqlScriptBo(conn, (incremental ? "" : "deploy_init_") + tenantId, tenant, tenantId, true));
            }
            return sqlScripts;
        }
        return sqlScripts;
    }


    public List<SQLScriptBo> listSqlScriptBo(Connection conn, String model, File dir, String tenantId, boolean recursion) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles;
        SchemaHistoryEntity entity = schemaHistoryService.queryLastVersion(conn, props, model);
        if (entity == null) {
            entity = schemaHistoryService.queryBaseLine(conn, props);
        }
        if (recursion) {
            try (Stream<Path> paths = java.nio.file.Files.walk(Paths.get(dir.getAbsolutePath()))) {
                sqlFiles = paths.map(Path::toFile).filter(f -> f.isFile() && f.getPath().endsWith(props.getSqlMigrationSuffix()))
                        .collect(Collectors.toList());
            }
        } else {
            File[] fs = dir.listFiles(f -> f.isFile() && f.getPath().endsWith(props.getSqlMigrationSuffix()));
            sqlFiles = Arrays.asList(fs == null ? new File[]{} : fs);
        }
        if (sqlFiles.size() == 0) {
            return new ArrayList<>(0);
        }
        int major = entity != null ? entity.getMajorVersion() : 0;
        int minor = entity != null ? entity.getMinorVersion() : 0;
        int patch = entity != null ? entity.getPatchVersion() : 0;
        sqlScripts = sqlFiles.stream().map(f -> {
            String[] p = f.getAbsolutePath().split("/|\\\\");
            if (
                // ddl 数据库脚本路径根据下的数据脚本model
                    "base".equals(model) ||
                            // dml 多租户数据脚本公共脚本特殊判断：1、 数据脚本初始化 2、数据脚本增量
                            ((("common_" + tenantId).equals(model) || ("deploy_init_common_" + tenantId).equals(model)) && "common".equals(p[p.length - 2])) ||
                            // dml 多租户数据脚本各个租户独立脚本特殊判断：数据脚本初始化
                            (("deploy_init_" + tenantId).equals(model) && tenantId != null && tenantId.equals(p[p.length - 2])) ||
                            // dml 多租户数据脚本各个租户独立脚本特殊判断：数据脚本增量
                            (tenantId != null && tenantId.equals(p[p.length - 2]))
                            // 正常情况 脚本根路径下无文件夹，数据脚本就在model dir下并在 mode 等于 dir name
                            || model.equals(p[p.length - 2])) {
                return new SQLScriptBo(props, model, f.getName(), f);
            } else {
                int i;
                if (tenantId != null && ("common_" + tenantId).equals(model)) {
                    i = Arrays.asList(p).indexOf("common");
                    p[i] = "common_" + tenantId;
                } else if (tenantId != null && ("deploy_init_common_" + tenantId).equals(model)) {
                    i = Arrays.asList(p).indexOf("common");
                    p[i] = "deploy_init_common_" + tenantId;
                } else {
                    i = Arrays.asList(p).indexOf(model);
                }
                if (i < 0) {
                    throw new IllegalArgumentException("脚本数据异常！");
                }
                String[] m = Arrays.copyOfRange(p, i, (p.length - 1));
                return new SQLScriptBo(props, String.join("_", m), f.getName(), f);
            }
        }).filter(bo -> bo.checkNeed(major, minor, patch))
                .sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    public void listSqlScriptBoCheck(List<SQLScriptBo> sqlScripts) {
        if (sqlScripts != null && sqlScripts.size() > 0) {
            Map<String, Long> versionGroup = sqlScripts.stream().collect(Collectors.groupingBy(b -> b.getModule() + ":" + b.getVersion(), Collectors.counting()));
            Optional<Map.Entry<String, Long>> error = versionGroup.entrySet().stream().filter(entry -> entry.getValue() > 1).findFirst();
            if (error.isPresent()) {
                String e = error.get().getKey();
                if (error.get().getKey().startsWith("common_")) {
                    e = "common" + e.substring(e.lastIndexOf(":"));
                }
                throw new IllegalArgumentException("数据脚本不符合规范：【 " + e + " 】下存在" + error.get().getValue() + "个版本相同的sql文件！");
            }
        }
    }

    public void increaseDbVersion(Connection conn, List<SQLScriptBo> sqlScripts, String tenantCol, String tenantVal) throws SQLException, IOException, NoSuchAlgorithmException, JSQLParserException {
        for (SQLScriptBo sqlScriptBo : sqlScripts) {
            this.increaseDbVersion(conn, sqlScriptBo, tenantCol, tenantVal);
        }
    }

    private void increaseDbVersion(Connection conn, SQLScriptBo sqlScriptBo, String tenantCol, String tenantVal) throws IOException, SQLException, NoSuchAlgorithmException, JSQLParserException {
        String checksum = Files.getFileChecksum(MessageDigest.getInstance("MD5"), sqlScriptBo.getFile());
        SchemaHistoryEntity entity = schemaHistoryService.queryByVersion(conn, props, sqlScriptBo.getModule(), sqlScriptBo.getVersion());
        boolean insertFlag = true;
        if (entity == null) {
            entity = new SchemaHistoryEntity();
            entity.setCreatedBy("auto");
        } else if (entity.getSuccess() == 1) {
            return;
        } else if (entity.getChecksum().equals(checksum)) {
            return;
        } else {
            insertFlag = false;
            entity.setUpdateBy("auto");
            entity.setUpdateTime(new Date());
        }
        entity.setType("SQL");
        entity.setModel(sqlScriptBo.getModule());
        entity.setScriptPath(sqlScriptBo.getFile().getPath());
        entity.setMajorVersion(sqlScriptBo.getMajorVersion());
        entity.setMinorVersion(sqlScriptBo.getMinorVersion());
        entity.setPatchVersion(sqlScriptBo.getPatchVersion());
        entity.setVersion(sqlScriptBo.getVersion());
        entity.setDescription(sqlScriptBo.getDescription());
        entity.setChecksum(checksum);
        log.debug("数据库[{}] 执行增量脚本:{}", conn.getMetaData().getURL(), sqlScriptBo.getFileName());
        // 开始时间
        LocalDateTime startTime = LocalDateTime.now();
        SuccessNum successNum = new SuccessNum();
        try {
            int startNum = 1;
            if (!insertFlag) {
                startNum = entity.getCodeBlockNum() + 1;
            }
            scriptRunner.runScript(conn, new FileReader(sqlScriptBo.getFile()), successNum, startNum, tenantCol, tenantVal);
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 成功 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            entity.setSuccess(1);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
        } catch (Exception e) {
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            entity.setSuccess(0);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 失败 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            log.error("", e);
            throw e;
        } finally {
            if (insertFlag) {
                this.schemaHistoryService.insertOne(conn, props, entity);
            } else {
                this.schemaHistoryService.updateOne(conn, props, entity);
            }
        }
    }
}
