package website.dachuan.migration.service.task;

import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.utils.DatabaseIdProvider;
import website.dachuan.migration.utils.SqlScriptCommon;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

@Slf4j
public class TenantDataScriptTask implements MigrationTask {
    private final MigrationProps props;
    private final SqlScriptCommon sqlScriptCommon;
    private final String tenantId;

    public TenantDataScriptTask(MigrationProps props, SqlScriptCommon sqlScriptCommon, String tenantId) {
        this.props = props;
        this.sqlScriptCommon = sqlScriptCommon;
        this.tenantId = tenantId;
    }

    @Override
    public Boolean doTask(Connection conn) throws SQLException, IOException, JSQLParserException, NoSuchAlgorithmException {
        log.debug("数据库[{}] TenantDataScriptTask doTask begin ...", conn.getMetaData().getURL());
        // 获取文件路径下的所有要执行的文件
        List<SQLScriptBo> scripts = this.createSqlScriptBo(conn);
        // 校验数据脚本文件编写方式是否正确
        sqlScriptCommon.listSqlScriptBoCheck(scripts);
        // 分步执行文件
        sqlScriptCommon.increaseDbVersion(conn, scripts, props.getTenantData().getTenantIdColumn(), tenantId);
        log.debug("数据库[{}] TenantDataScriptTask doTask end ...", conn.getMetaData().getURL());
        return true;
    }

    private List<SQLScriptBo> createSqlScriptBo(Connection conn) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts = new ArrayList<>();
        File folder;
        if (this.props.getTenantData().getScriptDirPath().startsWith("classpath:") || !this.props.getTenantData().getScriptDirPath().startsWith("/")) {
            URL url = this.getClass().getClassLoader().getResource(this.props.getTenantData().getScriptDirPath());
            if (url == null) {
                return sqlScripts;
            }
            folder = new File(url.getFile() + File.separator + DatabaseIdProvider.getDatabaseId(conn));
        } else {
            folder = new File(this.props.getTenantData().getScriptDirPath() + File.separator + DatabaseIdProvider.getDatabaseId(conn));
        }
        if (folder.exists()) {
            return sqlScriptCommon.listTenantDir(conn, folder, true, tenantId);
        }
        return sqlScripts;
    }
}
