package website.dachuan.migration.service.impl;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.dbutils.QueryRunner;
import org.apache.commons.dbutils.ResultSetHandler;
import org.apache.commons.dbutils.handlers.MapListHandler;
import website.dachuan.migration.bo.SuccessNum;
import website.dachuan.migration.mapping.DatabaseIdProvider;
import website.dachuan.migration.mapping.provider.OracleDatabaseId;
import website.dachuan.migration.parsesql.util.StringUtils;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.tool.SqlParserTenantUtil;
import website.dachuan.migration.utils.SqlScriptReader;

import java.io.Reader;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import java.util.Map;

/**
 * @author yqb22
 */
@Slf4j
public class SqlScriptRunner implements ISqlScriptRunner {
    private final QueryRunner runner;
    private final MyQueryRunner myQueryRunner;

    public SqlScriptRunner() {
        this.runner = new QueryRunner();
        this.myQueryRunner = new MyQueryRunner();
    }

    private static class MyQueryRunner {
        public void execute(Connection connection, String sql) throws SQLException {
            Statement statement = connection.createStatement();
            statement.execute(sql);
        }
    }

    @Override
    public List<Map<String, Object>> runScript(Connection conn, String sql, Object... param) throws SQLException {
        ResultSetHandler<List<Map<String, Object>>> rsh = new MapListHandler();
        return this.runner.query(conn, sql, rsh);
    }

    @Override
    public void runScript(Connection conn, Reader reader, SuccessNum successNum) throws SQLException {
        this.runScript(conn, reader, successNum, 1);
    }

    @Override
    public void runScript(Connection conn, Reader reader, SuccessNum successNum, int startNum) throws SQLException {
        if (startNum < 1) {
            startNum = 1;
        }
        String databaseProductName = DatabaseIdProvider.getDatabaseId(conn);
        List<String> sqlList = SqlScriptReader.readSqlScript(reader, databaseProductName);
        for (int i = (startNum - 1); i < sqlList.size(); i++) {
            String sql = sqlList.get(i);
            if (sql != null && !sql.isEmpty()) {
                if (new OracleDatabaseId().name().equals(DatabaseIdProvider.getDatabaseId(conn))) {
                    if (sql.contains("trigger") || sql.contains("TRIGGER")) {
                        myQueryRunner.execute(conn, sql);
                        continue;
                    }
                }
                if (sql.endsWith(";")) {
                    sql = sql.substring(0, sql.length() - 1);
                }
                runner.execute(conn, sql);
                successNum.setNum(i + 1);
            }
        }
    }

    @Override
    public void runScript(Connection conn, Reader reader, SuccessNum successNum, int startNum, String tenantCol, String tenantVal) throws SQLException {
        if (StringUtils.isEmpty(tenantCol) || StringUtils.isEmpty(tenantVal)) {
            this.runScript(conn, reader, successNum, startNum);
        } else {
            if (startNum < 1) {
                startNum = 1;
            }
            String databaseProductName = DatabaseIdProvider.getDatabaseId(conn);
            List<String> sqlList = SqlScriptReader.readSqlScript(reader, databaseProductName);
            for (int i = (startNum - 1); i < sqlList.size(); i++) {
                // 为sql 添加 租户信息
                String sql = SqlParserTenantUtil.addTenantInfo(sqlList.get(i), tenantCol, tenantVal);
                runner.execute(conn, sql);
                successNum.setNum(i + 1);
            }
        }
    }
}
