package website.dachuan.migration.service;

import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;

/**
 * @author yqb22
 */
public interface ISchemaHistoryService {
    /**
     * 查询最后一个版本的架构历史记录。
     *
     * @param conn  数据库连接，用于执行查询操作。
     * @param props 迁移属性，可能包含用于查询的特定属性或配置。
     * @return 返回一个包含架构历史记录的列表，这些记录代表了最后一个版本的架构。
     * @throws SQLException 如果在查询过程中遇到数据库错误，则抛出SQLException。
     */
    List<SchemaHistoryEntity> queryLastVersion(Connection conn, MigrationProps props) throws SQLException;

    /**
     * 查询模块下执行成功的版本最大的版本信息
     *
     * @param conn  数据库连接
     * @param props 配置属性
     * @param model 模块
     * @return 数据库版本信息
     * @throws SQLException 数据库异常
     */
    SchemaHistoryEntity queryLastVersion(Connection conn, MigrationProps props, String model) throws SQLException;

    /**
     * 查询某个版本sql执行情况
     *
     * @param conn    数据库连接
     * @param props   配置属性
     * @param model   模块
     * @param version 版本信息
     * @return 数据库版本信息
     * @throws SQLException 数据库异常
     */
    SchemaHistoryEntity queryByVersion(Connection conn, MigrationProps props, String model, String version) throws SQLException;

    /**
     * 查询基础线信息。
     *
     * @param conn  数据库连接，用于执行数据库操作。
     * @param props 迁移属性，包含必要的迁移配置信息。
     * @return SchemaHistoryEntity 实例，包含基础线的架构历史信息。
     * @throws SQLException 如果执行数据库操作时出现错误，则抛出SQLException。
     */
    SchemaHistoryEntity queryBaseLine(Connection conn, MigrationProps props) throws SQLException;

    /**
     * 查询数据库初始化脚本执行情况。
     *
     * @param conn  数据库连接，用于执行数据库操作。
     * @param model 模块
     * @param props 迁移属性，包含必要的迁移配置信息。
     * @return list SchemaHistoryEntity 实例，包含基础线的架构历史信息。
     * @throws SQLException 如果执行数据库操作时出现错误，则抛出SQLException。
     */
    List<SchemaHistoryEntity> queryDeployInit(Connection conn, String model, MigrationProps props) throws SQLException;

    /**
     * 向数据库中插入一条记录。
     *
     * @param conn   数据库连接，用于执行插入操作。
     * @param props  迁移属性，包含迁移相关的配置信息。
     * @param entity 要插入的数据实体，包含 schema 历史信息。
     * @throws SQLException 如果插入操作中发生 SQL 错误。
     */
    void insertOne(Connection conn, MigrationProps props, SchemaHistoryEntity entity) throws SQLException;

    /**
     * 更新数据库中的一个Schema历史记录。
     *
     * @param conn   数据库连接，用于执行更新操作。
     * @param props  迁移属性，包含迁移相关的配置信息。
     * @param entity 要更新的Schema历史实体对象。
     * @return 更新操作影响的行数，通常为1表示更新了一行，若未更新则为0。
     * @throws SQLException 如果执行更新操作时发生数据库错误。
     */
    int updateOne(Connection conn, MigrationProps props, SchemaHistoryEntity entity) throws SQLException;
}

