package website.dachuan.migration;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.IDBOperation;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.service.task.CreateHistoryTableTask;
import website.dachuan.migration.service.task.DeployInitTableTask;
import website.dachuan.migration.service.task.IncreaseVersionTask;
import website.dachuan.migration.service.task.InsertBaselineTask;
import website.dachuan.migration.service.task.handles.TaskHandle;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

import static website.dachuan.migration.commons.Constants.TYPE_BASE_LINE;

@Slf4j
class MigrationProcess {
    private final MigrationProps props;
    private final IDBOperation dbOperation;
    private final ISqlScriptRunner scriptRunner;
    private final ISchemaHistoryService schemaHistoryService;
    private final TaskHandle taskHandle;
    /**
     * 数据库版本控制任务队列
     */
    private final BlockingQueue<MigrationTask> tasks = new LinkedBlockingQueue<>();

    public MigrationProcess(MigrationProps props, IDBOperation dbOperation, ISqlScriptRunner scriptRunner, TaskHandle taskHandle,
                            ISchemaHistoryService schemaHistoryService) {
        this.props = props;
        this.dbOperation = dbOperation;
        this.scriptRunner = scriptRunner;
        this.taskHandle = taskHandle;
        this.schemaHistoryService = schemaHistoryService;
    }

    public void exec(Connection conn) throws Exception {
        OperationMode operationMode = this.chargeOperationMode(conn);
        assert operationMode != null;
        this.assemblyTaskChain(operationMode);
        while (!this.tasks.isEmpty()) {
            this.tasks.poll().doTask(conn);
        }
    }

    private void assemblyTaskChain(OperationMode operationMode) {
        switch (operationMode) {
            case DEPLOY_INIT_NO_HISTORY_TABLE:
                tasks.add(new CreateHistoryTableTask(props, scriptRunner));
                tasks.add(new DeployInitTableTask(props, taskHandle));
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            case DEPLOY_INIT:
                tasks.add(new DeployInitTableTask(props, taskHandle));
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            case BASELINE_CREATE:
                tasks.add(new CreateHistoryTableTask(props, scriptRunner));
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            case BASELINE_INIT:
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            default:
                break;
        }
        tasks.add(new IncreaseVersionTask(props, taskHandle));
    }

    private OperationMode chargeOperationMode(Connection conn) throws SQLException {
        // 判断当前database是否非空
        List<String> tbNames = dbOperation.listTables(conn);
        // 当前database为空，首次启动服务，导入全部数据库脚本，并创建数据库版本控制表，并生成数据库版本记录。
        if (tbNames.size() == 0) {
            return OperationMode.DEPLOY_INIT_NO_HISTORY_TABLE;
        }
        // 当前数据库只存在版本控制表并且无基线数据
        if (tbNames.size() == 1 && tbNames.get(0).toLowerCase().equals(props.getSchemaHistoryTableName())) {
            return OperationMode.DEPLOY_INIT;
        }
        // 如果当前database非空，但还没有数据库版本控制表,创建数据库版本控制表，生成baseline记录；然后做增量的sql脚本执行。
        if (tbNames.stream().map(String::toLowerCase).noneMatch(props.getSchemaHistoryTableName()::equals)) {
            return OperationMode.BASELINE_CREATE;
        }
        // 查询版本记录表是否存在数据
        List<SchemaHistoryEntity> historyList = schemaHistoryService.queryLastVersion(conn, props);
        // 已经存在数据库版本控制表，但是没有版本记录信息，生成baseline记录；然后做增量的sql脚本执行。
        if (historyList == null || historyList.size() == 0) {
            return OperationMode.BASELINE_INIT;
            // 存在版本记录但是无基线数据，说明初始化脚本执行过程中出错
        } else if (historyList.stream().noneMatch(s -> TYPE_BASE_LINE.equals(s.getType()))) {
            return OperationMode.DEPLOY_INIT;
        } else {
            // 已经存在数据库版本控制表，并且存在执行记录，做增量的sql脚本执行。
            return OperationMode.DEPLOY_INCREASE;
        }
    }

    public enum OperationMode {
        // 项目首次部署，数据库没有任何表。
        // 该操作会生成数据库版本控制表，执行数据库初始化脚本，更新数据库版本控制表数据。
        DEPLOY_INIT_NO_HISTORY_TABLE,

        // 项目部署，数据库中存在版本控制表但是无业务表没有任何表。
        // 执行数据库初始化脚本，更新数据库版本控制表数据。
        DEPLOY_INIT,

        // 项目增量部署，之前已经导入业务表与数据库版本控制表。
        // 该操作根据已有的数据库版本控制表中的记录判断哪些脚本需要执行，然后执行脚本并插入新的数据库版本记录。
        DEPLOY_INCREASE,

        // 一个已经上线的项目初次使用数据库版本控制，之前已经导入业务表，但没有数据库版本控制表。
        // 该操作会创建数据库版本控制表，并写入一条版本基线记录，然后基于属性配置的基线版本确定哪些脚本需要执行。
        // 执行脚本后向数据库版本控制表插入新的版本记录。
        BASELINE_CREATE,

        // 一个已经上线的项目初次使用数据库版本控制，之前已经导入业务表并生成数据库版本控制表，但数据库版本控制表没有基线数据。
        // 该操作会写入一条版本基线记录，然后基于属性配置的基线版本确定哪些脚本需要执行。
        // 执行脚本后向数据库版本控制表插入新的版本记录。
        BASELINE_INIT
    }
}
