package website.dachuan.migration.utils;

import lombok.extern.slf4j.Slf4j;
import sun.net.www.protocol.file.FileURLConnection;

import java.io.*;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.util.*;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author yqb22
 */
@Slf4j
public class Files {
    private static File tempFile;

    public static String getFileChecksum(MessageDigest digest, File file) throws IOException {
        FileInputStream fis = new FileInputStream(file);
        byte[] byteArray = new byte[1024];
        int bytesCount;
        while ((bytesCount = fis.read(byteArray)) != -1) {
            digest.update(byteArray, 0, bytesCount);
        }
        fis.close();
        byte[] bytes = digest.digest();
        StringBuilder sb = new StringBuilder();
        for (byte aByte : bytes) {
            sb.append(Integer.toString((aByte & 0xff) + 0x100, 16).substring(1));
        }
        return sb.toString();
    }

    public static List<File> getAllFiles(File root, boolean recursion) throws IOException {
        return Files.getAllFiles(root, recursion, null);
    }

    /**
     * 获取文件夹下所有文件<p>可以筛选某一后缀的文件
     *
     * @param root      文件夹
     * @param recursion 是否递归遍历下级文件夹下的所有文件
     * @param suffix    文件名称的后缀 <code>something.sql --> .sql</code>
     * @return 沒有文件时返回空list，有文件时返回文件list
     */
    public static List<File> getAllFiles(File root, boolean recursion, String suffix) throws IOException {
        List<File> files;
        if (root == null) {
            return new ArrayList<>(0);
        }
        if (recursion) {
            try (Stream<Path> paths = java.nio.file.Files.walk(Paths.get(root.getAbsolutePath()))) {
                files = paths.map(Path::toFile).filter(File::isFile).filter(f -> suffix == null || (suffix.length() > 0 && f.getPath().endsWith(suffix))).collect(Collectors.toList());
            }
        } else {
            File[] fs = root.listFiles(f -> f.isFile() && (suffix == null || (suffix.length() > 0 && f.getPath().endsWith(suffix))));
            files = Arrays.asList(fs == null ? new File[]{} : fs);
        }
        if (files.size() == 0) {
            return new ArrayList<>(0);
        }
        return files;
    }

    /**
     * copy 文件夹下文件或者jar包中文件夹
     *
     * @param urls       文件夹url的集合
     * @param targetFile 目标文件夹
     * @throws IOException io e
     */
    public static void loadRecourse(List<URL> urls, File targetFile) throws IOException {
        for (URL url : urls) {
            loadRecourse(url, targetFile);
        }
    }

    /**
     * copy 文件夹下文件或者jar包中文件夹
     *
     * @param url        文件夹url
     * @param targetFile 目标文件夹
     * @throws IOException io e
     */
    public static void loadRecourse(URL url, File targetFile) throws IOException {
        URLConnection urlConnection = url.openConnection();
        if (urlConnection instanceof FileURLConnection) {
            String decodePath = URLDecoder.decode(url.getPath(), "utf-8");
            copyFileResources(new File(decodePath), new File(targetFile, getDirName(url.getFile())));
        } else if (urlConnection instanceof JarURLConnection) {
            copyJarResources((JarURLConnection) urlConnection, targetFile);
        }
    }

    /**
     * 当前运行环境资源文件是在jar里面的
     *
     * @param jarURLConnection 文件url
     * @throws IOException io e
     */
    public static void copyJarResources(JarURLConnection jarURLConnection, File targetFile) throws IOException {
        ///计算复制开始的时间
        long begin = System.currentTimeMillis();
        JarFile jarFile = jarURLConnection.getJarFile();
        Enumeration<JarEntry> entries = jarFile.entries();
        String rootDir = jarURLConnection.getEntryName();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            if (rootDir != null && rootDir.length() > 0) {
                if (entry.getName().startsWith(rootDir)) {
                    loadRecourseFromJar(entry, rootDir, targetFile);
                }
            } else {
                loadRecourseFromJar(entry, null, targetFile);
            }
        }
        jarFile.close();
        //计算复制完成的时间
        long end = System.currentTimeMillis();
        log.debug("jar:{}中文件夹：{}复制到文件：{}完成，耗时：{}分钟。", jarFile.getName(), rootDir, targetFile.getName(), ((end - begin) / 1000) / 60);
    }

    /**
     * 将 jar中文件复制到本地
     * <br>在复制到新文件时只会复制跟路径及其下层的文件夹以及文件；
     * <br>举例：根路径：META-INF/org/springframework/boot，则只会复制boot及其下的文件夹与文件，文件结构如下:
     * <br>boot
     * <br>&ensp;&ensp;--XXX
     * <br>&ensp;&ensp;&ensp;&ensp;XXX123.sql
     * <br>
     *
     * @param jarEntry   jarEntry
     * @param rootPath   需要复制的文件夹路径
     * @param targetFile 目标文件夹
     */
    public static void loadRecourseFromJar(JarEntry jarEntry, String rootPath, File targetFile) throws IOException {
        if (!targetFile.exists()) {
            if (!targetFile.mkdirs()) {
                throw new IOException("文件夹创建失败！文件夹：" + targetFile.getAbsolutePath());
            }
        }
        String dirName = null;
        if (rootPath != null && rootPath.length() > 0) {
            String[] sa = rootPath.split("/");
            if (sa.length > 0) {
                dirName = sa[sa.length - 1];
            }
        }
        String fileName;
        if ((rootPath == null || rootPath.length() == 0)) {
            fileName = jarEntry.getName();
        } else if (dirName == null || dirName.length() == 0) {
            fileName = jarEntry.getName().replace(rootPath, "");
        } else {
            fileName = dirName + "/" + jarEntry.getName().replace(rootPath, "");
        }

        if (fileName.startsWith("/")) {
            fileName = fileName.substring(1);
        }
        if (jarEntry.isDirectory()) {
            File f = new File(targetFile, fileName);
            if (!f.exists()) {
                if (!f.mkdirs()) {
                    throw new IOException("文件夹创建失败！文件夹：" + targetFile.getAbsolutePath());
                }
            }
        } else {
            File file = new File(targetFile, fileName);
            byte[] buffer = new byte[1024];
            int readBytes;
            try (InputStream is = Files.class.getClassLoader().getResourceAsStream(jarEntry.getName()); OutputStream os = new FileOutputStream(file)) {
                if (is != null) {
                    while ((readBytes = is.read(buffer)) != -1) {
                        os.write(buffer, 0, readBytes);
                    }
                }
            } catch (IOException e) {
                log.error(log.getName(), e);
            }
        }
    }

    /**
     * 当前运行环境资源文件是在文件里面的
     *
     * @param file       需要复制的文件夹
     * @param targetFile 目标文件
     */
    public static void copyFileResources(File file, File targetFile) throws IOException {
        ///计算复制开始的时间
        long begin = System.currentTimeMillis();
        if (!file.isDirectory()) {
            return;
        }
        if (!targetFile.exists()) {
            boolean t = targetFile.mkdirs();
            if (!t) {
                throw new IOException("文件夹创建失败！文件夹：" + targetFile.getAbsolutePath());
            }
        }
        File[] files = file.listFiles();
        if (files != null && files.length > 0) {
            for (File f : files) {
                if (f.isDirectory()) {
                    copyFileResources(f, new File(targetFile, f.getName()));
                } else if (f.isFile()) {
                    copyFile(f, new File(targetFile.getPath(), f.getName()));
                }
            }
        }
        //计算复制完成的时间
        long end = System.currentTimeMillis();
        log.debug("文件夹：{}复制到文件：{}完成，耗时：{}秒。", file.getName(), targetFile.getName(), ((end - begin) / 1000));
    }

    /**
     * 复制文件
     *
     * @param file       原文件
     * @param targetFile 复制后的文件
     */
    public static void copyFile(File file, File targetFile) {
        try (FileInputStream fis = new FileInputStream(file); FileOutputStream fos = new FileOutputStream(targetFile)) {
            byte[] bys = new byte[1024];
            int len;
            while ((len = fis.read(bys)) != -1) {
                fos.write(bys, 0, len);
            }
        } catch (IOException e) {
            log.error(log.getName(), e);
        }
    }

    /**
     * 复制文件路径下的文件到 temp文件夹下<p>
     * 如果在文件系统中可以获取到就是文件系统中文件如果文件系统中获取不到就是 class path 下相对路径文件夹
     *
     * @param path 相对路径
     * @throws IOException io异常
     */
    public static void copyFile(String path) throws IOException {
        List<URL> urls = new ArrayList<>();
        File f = new File(path);
        if (!f.exists()) {
            Enumeration<URL> resources = Files.class.getClassLoader().getResources(path);
            while (resources.hasMoreElements()) {
                URL resourceUrl = resources.nextElement();
                urls.add(resourceUrl);
            }
        } else {
            urls.add(f.toURI().toURL());
        }
        if (urls.size() != 0) {
            Files.loadRecourse(urls, Files.createTemp());
        }
    }

    /**
     * 先根遍历序递归删除文件夹
     *
     * @param dirFile 要被删除的文件或者目录
     * @return 删除成功返回true, 否则返回false
     */
    public static boolean delete(File dirFile) {
        // 如果dir对应的文件不存在，则退出
        if (!dirFile.exists()) {
            return false;
        }

        if (dirFile.isFile()) {
            return dirFile.delete();
        } else {
            File[] files = dirFile.listFiles();
            if (files != null && files.length > 0) {
                for (File file : Objects.requireNonNull(dirFile.listFiles())) {
                    delete(file);
                }
            }
        }
        return dirFile.delete();
    }

    /**
     * 创建临时文件夹
     *
     * @return 夹
     * @throws IOException io e
     */
    public static File createTemp() throws IOException {
        if (tempFile == null) {
            tempFile = java.nio.file.Files.createTempDirectory("migration_").toFile();
            log.debug("migration temp dir path is : " + tempFile.getAbsolutePath());
        }
        return tempFile;
    }

    public static void cleanTemp() throws IOException {
        if (tempFile == null) {
            tempFile = java.nio.file.Files.createTempDirectory("migration_").toFile();
            log.debug("migration temp dir path is : " + tempFile.getAbsolutePath());
        }
        File p = tempFile.getParentFile();
        if (p.isDirectory()) {
            List<File> temps = Arrays.stream(Objects.requireNonNull(p.listFiles())).filter(f -> f.isDirectory() && f.getName().startsWith("migration_")).collect(Collectors.toList());
            temps.forEach(Files::delete);
        }
        tempFile = null;
    }

    public static File getTemp() {
        if (tempFile == null) {
            return null;
        }
        return tempFile;
    }

    public static boolean deleteTemp() {
        if (delete(tempFile)) {
            tempFile = null;
            log.debug("migration 临时文件删除成功！");
            return true;
        }
        return false;
    }

    public static String getDirName(String path) {
        if (path == null || path.length() == 0) {
            throw new IllegalArgumentException("path 不能为空！");
        }
        String dn;
        if (path.contains("/")) {
            String[] dns = path.split("/");
            dn = dns[dns.length - 1];
        } else if (path.contains("\\")) {
            String[] dns = path.split("\\\\");
            dn = dns[dns.length - 1];
        } else {
            dn = path;
        }
        if (dn.contains(".")) {
            throw new IllegalArgumentException("path 不能为文件夹！");
        }
        return dn;
    }
}
