package website.dachuan.migration.service.task.handles;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.bo.SuccessNum;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.utils.Files;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static website.dachuan.migration.commons.Constants.*;

/**
 * @author yqb22
 */
@Slf4j
public class TaskHandle {
    private final ISchemaHistoryService schemaHistoryService;
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;

    public TaskHandle(MigrationProps props, ISchemaHistoryService schemaHistoryService, ISqlScriptRunner scriptRunner) {
        this.schemaHistoryService = schemaHistoryService;
        this.props = props;
        this.scriptRunner = scriptRunner;
    }

    /**
     * 脚本路径规范检测
     *
     * @param sqlScripts 脚本对象
     */
    public void listSqlScriptBoCheck(List<SQLScriptBo> sqlScripts) {
        if (sqlScripts != null && sqlScripts.size() > 0) {
            Map<String, Long> versionGroup = sqlScripts.stream().collect(Collectors.groupingBy(b -> b.getModule() + ":" + b.getVersion(), Collectors.counting()));
            Optional<Map.Entry<String, Long>> error = versionGroup.entrySet().stream().filter(entry -> entry.getValue() > 1).findFirst();
            if (error.isPresent()) {
                String e = error.get().getKey();
                if (error.get().getKey().startsWith(COMMON + UNDER_LINE)) {
                    e = COMMON + e.substring(e.lastIndexOf(":"));
                }
                throw new IllegalArgumentException("数据脚本不符合规范：【 " + e + " 】下存在" + error.get().getValue() + "个版本相同的sql文件！");
            }
        }
    }

    /**
     * 根据版本信息检索需要执行的增量脚本
     *
     * @param conn      数据源连接
     * @param model     模块
     * @param dir       脚本文件夹
     * @param recursion 是否遍历下级文件夹
     * @return sql bo
     * @throws IOException  IOException
     * @throws SQLException SQLException
     */
    public List<SQLScriptBo> listSqlScriptBo(Connection conn, String model, File dir, boolean recursion) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles = Files.getAllFiles(dir, recursion, props.getSqlMigrationSuffix());
        SchemaHistoryEntity entity = schemaHistoryService.queryLastVersion(conn, props, model);
        if (entity == null) {
            entity = schemaHistoryService.queryBaseLine(conn, props);
        }
        int major = entity != null ? entity.getMajorVersion() : 0;
        int minor = entity != null ? entity.getMinorVersion() : 0;
        int patch = entity != null ? entity.getPatchVersion() : 0;
        sqlScripts = sqlFiles.stream().map(f -> {
            int i;
            String[] p = f.getAbsolutePath().split("/|\\\\");
            // ddl 数据库脚本路径根据下的数据脚本model
            if (BASE.equals(model)) {
                return new SQLScriptBo(props, model, f.getName(), f);
            }
            if (model.startsWith(COMMON)) {
                i = Arrays.asList(p).indexOf(COMMON);
                p[i] = model;
            } else {
                i = Arrays.asList(p).indexOf(model);
            }
            if (i < 0) {
                throw new IllegalArgumentException("脚本数据异常！");
            }
            String[] m = Arrays.copyOfRange(p, i, (p.length - 1));
            return new SQLScriptBo(props, String.join(UNDER_LINE, m), f.getName(), f);
        }).filter(bo -> bo.checkNeed(major, minor, patch))
                .sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    /**
     * 检索需要执行的初始化脚本
     *
     * @param conn      数据源连接
     * @param props     参数配置
     * @param model     模块
     * @param dir       脚本文件夹
     * @param recursion 是否遍历下级文件夹
     * @return sql bo
     * @throws IOException  IOException
     * @throws SQLException SQLException
     */
    public List<SQLScriptBo> listDeployInitSqlScriptBo(Connection conn, MigrationProps props, String model, File dir, boolean recursion) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles = Files.getAllFiles(dir, recursion, props.getSqlMigrationSuffix());
        List<SchemaHistoryEntity> entities = schemaHistoryService.queryDeployInit(conn, model, props);
        sqlScripts = sqlFiles.stream().map(f -> {
            String[] p = f.getAbsolutePath().split("/|\\\\");
            if (entities != null && entities.size() > 0) {
                for (SchemaHistoryEntity schemaHistory : entities) {
                    String[] ep = schemaHistory.getScriptPath().split("/|\\\\");
                    if (ep[ep.length - 1].equals(p[p.length - 1]) && schemaHistory.getSuccess() == 1) {
                        return null;
                    }
                }
            }
            return new SQLScriptBo(props, model, f.getName(), f);
        }).filter(Objects::nonNull).sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    public void increaseDbVersion(Connection conn, List<SQLScriptBo> sqlScripts, String tenantCol, String tenantVal) throws SQLException, IOException, NoSuchAlgorithmException {
        for (SQLScriptBo sqlScriptBo : sqlScripts) {
            this.increaseDbVersion(conn, sqlScriptBo, tenantCol, tenantVal);
        }
    }

    private void increaseDbVersion(Connection conn, SQLScriptBo sqlScriptBo, String tenantCol, String tenantVal) throws IOException, SQLException, NoSuchAlgorithmException {
        if (Files.getTemp() == null) {
            throw new IOException("临时文件获取异常！");
        }
        String checksum = Files.getFileChecksum(MessageDigest.getInstance("MD5"), sqlScriptBo.getFile());
        // 支持初始化脚本同一文件夹下多个脚本且没有版本号的情况
        SchemaHistoryEntity entity = schemaHistoryService.queryByVersion(conn, props, sqlScriptBo.getModule(), sqlScriptBo.getVersion());
        boolean insertFlag = true;
        if (entity == null) {
            entity = new SchemaHistoryEntity();
            entity.setCreatedBy("auto");
        } else if (entity.getSuccess() == 1) {
            return;
        } else if (entity.getChecksum().equals(checksum)) {
            return;
        } else {
            insertFlag = false;
            entity.setUpdateBy("auto");
        }
        entity.setType(TYPE_SQL);
        entity.setModel(sqlScriptBo.getModule());
        entity.setScriptPath(sqlScriptBo.getFile().getPath().replace(Files.getTemp().getAbsolutePath() + File.separator, ""));
        entity.setMajorVersion(sqlScriptBo.getMajorVersion());
        entity.setMinorVersion(sqlScriptBo.getMinorVersion());
        entity.setPatchVersion(sqlScriptBo.getPatchVersion());
        entity.setVersion(sqlScriptBo.getVersion());
        entity.setDescription(sqlScriptBo.getDescription());
        entity.setChecksum(checksum);
        log.debug("数据库[{}] 执行增量脚本:{}", conn.getMetaData().getURL(), sqlScriptBo.getFileName());
        // 开始时间
        LocalDateTime startTime = LocalDateTime.now();
        SuccessNum successNum = new SuccessNum();
        try {
            int startNum = 1;
            if (!insertFlag) {
                startNum = entity.getCodeBlockNum() + 1;
            }
            // 解决在jar包下执行sql中有中文乱码的问题
            FileInputStream fis = new FileInputStream(sqlScriptBo.getFile());
            InputStreamReader isr = new InputStreamReader(fis, StandardCharsets.UTF_8);
            scriptRunner.runScript(conn, isr, successNum, startNum, tenantCol, tenantVal);
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 成功 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            entity.setSuccess(1);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
        } catch (Exception e) {
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            entity.setSuccess(0);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 失败 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            log.error("", e);
            throw e;
        } finally {
            if (insertFlag) {
                this.schemaHistoryService.insertOne(conn, props, entity);
            } else {
                this.schemaHistoryService.updateOne(conn, props, entity);
            }
        }
    }
}
