package website.dachuan.migration.service.task;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.dao.sql.SqlProviderStrategy;
import website.dachuan.migration.parsesql.util.StringUtils;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.service.task.handles.TaskHandle;
import website.dachuan.migration.utils.Files;

import java.io.File;
import java.io.IOException;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

import static website.dachuan.migration.commons.Constants.*;

/**
 * @author yqb22
 */
@Slf4j
public class TenantDataScriptTask implements MigrationTask {
    private final MigrationProps props;
    private final String tenantId;
    private final TaskHandle taskHandle;
    private final boolean isPrimary;

    public TenantDataScriptTask(MigrationProps props, TaskHandle taskHandle, String tenantId, boolean isPrimary) {
        this.props = props;
        this.taskHandle = taskHandle;
        this.tenantId = tenantId;
        this.isPrimary = isPrimary;
    }

    @Override
    public Boolean doTask(Connection conn) throws SQLException, IOException, NoSuchAlgorithmException {
        log.debug("数据库[{}] TenantDataScriptTask doTask begin ...", conn.getMetaData().getURL());
        // 获取文件路径下的所有要执行的文件
        List<SQLScriptBo> scripts = this.createSqlScriptBo(conn);
        // 校验数据脚本文件编写方式是否正确
        taskHandle.listSqlScriptBoCheck(scripts);
        // 分步执行文件
        taskHandle.increaseDbVersion(conn, scripts, props.getTenantData().getTenantIdColumn(), tenantId);
        log.debug("数据库[{}] TenantDataScriptTask doTask end ...", conn.getMetaData().getURL());
        return true;
    }

    private List<SQLScriptBo> createSqlScriptBo(Connection conn) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts = new ArrayList<>();
        File tempDir = Files.getTemp();
        if (tempDir == null) {
            return sqlScripts;
        }
        File folder = new File(tempDir.getAbsolutePath() + File.separator + Files.getDirName(this.props.getTenantData().getScriptDirPath()) + File.separator + SqlProviderStrategy.getSqlProvider(conn).sqlDirName());
        if (folder.exists()) {
            File[] files = folder.listFiles();
            if (files == null || files.length == 0) {
                return sqlScripts;
            }
            if (isPrimary) {
                // 检索超级租户文件夹下sql文件
                File st = new File(folder.getAbsolutePath() + File.separator + SUPER_TENANT_DIR);
                if (st.exists()) {
                    sqlScripts.addAll(this.taskHandle.listSqlScriptBo(conn, SUPER_TENANT_DIR, st, true));
                }
            }
            if (StringUtils.isEmpty(tenantId)) {
                return sqlScripts;
            }
            // 检索公共文件夹下sql文件
            File common = new File(folder.getAbsolutePath() + File.separator + COMMON);
            if (common.exists()) {
                sqlScripts.addAll(this.taskHandle.listSqlScriptBo(conn, COMMON + UNDER_LINE + tenantId, common, true));
            }
            // 检索租户文件夹下文件
            File tenant = new File(folder.getAbsolutePath() + File.separator + tenantId);
            if (tenant.exists()) {
                sqlScripts.addAll(this.taskHandle.listSqlScriptBo(conn, tenantId, tenant, true));
            }
            return sqlScripts;
        }
        return sqlScripts;
    }
}
