package website.dachuan.migration.mapping;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.parsesql.Parser;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;
import java.util.ServiceLoader;
import java.util.concurrent.ConcurrentHashMap;

@Slf4j
public class DatabaseIdProvider {
    private static final Map<String, DatabaseId> databaseIdMap = new ConcurrentHashMap<>();

    static {
        ServiceLoader<DatabaseId> databaseIds = ServiceLoader.load(DatabaseId.class);
        for (DatabaseId databaseId : databaseIds) {
            databaseIdMap.put(databaseId.name(), databaseId);
        }
    }

    public static String getDatabaseId(Connection connection) {
        return getDatabaseIdEnum(connection).name();
    }

    public static boolean getDatabaseCanAutoIncrement(Connection connection) {
        return getDatabaseIdEnum(connection).autoIncrement();
    }

    public static DatabaseId valueOf(Connection conn) {
        String databaseProductName = getDatabaseId(conn);
        return DatabaseIdProvider.valueOf(databaseProductName);
    }

    public static DatabaseId valueOf(String name) {
        return databaseIdMap.entrySet().stream().filter(e -> e.getKey().equals(name)).map(Map.Entry::getValue).findFirst().orElse(null);
    }

    public static DatabaseId valueOfUk(String uk) {
        return databaseIdMap.values().stream().filter(e -> e.uk().equals(uk)).findFirst().orElse(null);
    }

    static DatabaseId getDatabaseIdEnum(Connection connection) {
        if (connection == null) {
            throw new NullPointerException("DatabaseIdProvider connection cannot be null");
        } else {
            try {
                String productName = connection.getMetaData().getDatabaseProductName();
                String url = connection.getMetaData().getURL();
                if (url.toLowerCase().contains(productName.toLowerCase())) {
                    return DatabaseIdProvider.valueOf(productName);
                } else {
                    String[] urls = url.split(":");
                    return DatabaseIdProvider.valueOfUk(urls[1]);
                }
            } catch (SQLException e) {
                log.error(log.getName(), e);
                return new DatabaseId() {
                    @Override
                    public String name() {
                        return "NnKnown";
                    }

                    @Override
                    public String uk() {
                        return "unknown";
                    }

                    @Override
                    public boolean ddlTransaction() {
                        return false;
                    }

                    @Override
                    public boolean autoIncrement() {
                        return false;
                    }

                    @Override
                    public Parser getSqlParser() {
                        return null;
                    }
                };
            }
        }
    }
}
