package website.dachuan.migration.utils;

import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.bo.SuccessNum;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

import static website.dachuan.migration.commons.Constants.*;

/**
 * @author yqb22
 */
@Slf4j
public class SqlScriptCommon {
    private final ISchemaHistoryService schemaHistoryService;
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;

    public SqlScriptCommon(MigrationProps props, ISchemaHistoryService schemaHistoryService, ISqlScriptRunner scriptRunner) {
        this.schemaHistoryService = schemaHistoryService;
        this.props = props;
        this.scriptRunner = scriptRunner;
    }

    public List<SQLScriptBo> listTenantDir(Connection conn, File folder, boolean incremental, String tenantId) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts = new ArrayList<>();
        if (folder.length() > 0) {
            // 检索公用文件夹下文件
            File common = new File(folder.getAbsolutePath() + File.separator + COMMON);
            if (common.exists()) {
                sqlScripts.addAll(this.listSqlScriptBo(conn, (incremental ? COMMON + UNDER_LINE : DEPLOY_INIT + UNDER_LINE + COMMON + UNDER_LINE) + tenantId, common, tenantId, true));
            }
            // 检索租户文件夹下文件
            File tenant = new File(folder.getAbsolutePath() + File.separator + tenantId);
            if (tenant.exists()) {
                sqlScripts.addAll(this.listSqlScriptBo(conn, (incremental ? "" : DEPLOY_INIT + UNDER_LINE) + tenantId, tenant, tenantId, true));
            }
            return sqlScripts;
        }
        return sqlScripts;
    }

    public List<SQLScriptBo> listSqlScriptBo(Connection conn, String model, File dir, String tenantId, boolean recursion) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles = Files.getAllFiles(dir, recursion, props.getSqlMigrationSuffix());
        SchemaHistoryEntity entity = schemaHistoryService.queryLastVersion(conn, props, model);
        if (entity == null) {
            entity = schemaHistoryService.queryBaseLine(conn, props);
        }
        int major = entity != null ? entity.getMajorVersion() : 0;
        int minor = entity != null ? entity.getMinorVersion() : 0;
        int patch = entity != null ? entity.getPatchVersion() : 0;
        sqlScripts = sqlFiles.stream().map(f -> {
            String[] p = f.getAbsolutePath().split("/|\\\\");
            if (
                // ddl 数据库脚本路径根据下的数据脚本model
                    BASE.equals(model) ||
                            // dml 多租户数据脚本公共脚本特殊判断：1、 数据脚本初始化 2、数据脚本增量
                            (((COMMON + UNDER_LINE + tenantId).equals(model) || (DEPLOY_INIT + UNDER_LINE + COMMON + UNDER_LINE + tenantId).equals(model)) && COMMON.equals(p[p.length - 2])) ||
                            // dml 多租户数据脚本各个租户独立脚本特殊判断：数据脚本初始化
                            ((DEPLOY_INIT + UNDER_LINE + tenantId).equals(model) && tenantId != null && tenantId.equals(p[p.length - 2])) ||
                            // dml 多租户数据脚本各个租户独立脚本特殊判断：数据脚本增量
                            (tenantId != null && tenantId.equals(p[p.length - 2]))
                            // 正常情况 脚本根路径下无文件夹，数据脚本就在model dir下并在 mode 等于 dir name
                            || model.equals(p[p.length - 2])) {
                return new SQLScriptBo(props, model, f.getName(), f);
            } else {
                int i;
                if (tenantId != null && (COMMON + UNDER_LINE + tenantId).equals(model)) {
                    i = Arrays.asList(p).indexOf(COMMON);
                    p[i] = COMMON + UNDER_LINE + tenantId;
                } else if (tenantId != null && (DEPLOY_INIT + UNDER_LINE + COMMON + UNDER_LINE + tenantId).equals(model)) {
                    i = Arrays.asList(p).indexOf(COMMON);
                    p[i] = DEPLOY_INIT + UNDER_LINE + COMMON + UNDER_LINE + tenantId;
                } else {
                    i = Arrays.asList(p).indexOf(model);
                }
                if (i < 0) {
                    throw new IllegalArgumentException("脚本数据异常！");
                }
                String[] m = Arrays.copyOfRange(p, i, (p.length - 1));
                return new SQLScriptBo(props, String.join(UNDER_LINE, m), f.getName(), f);
            }
        }).filter(bo -> bo.checkNeed(major, minor, patch))
                .sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    public List<SQLScriptBo> listDeployInitSqlScriptBo(Connection conn, String model, File dir, boolean recursion) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles = Files.getAllFiles(dir, recursion, props.getSqlMigrationSuffix());
        List<SchemaHistoryEntity> entities = schemaHistoryService.queryDeployInit(conn, model, props);
        sqlScripts = sqlFiles.stream().map(f -> {
            String[] p = f.getAbsolutePath().split("/|\\\\");
            if (entities != null && entities.size() > 0) {
                for (SchemaHistoryEntity schemaHistory : entities) {
                    String[] ep = schemaHistory.getScriptPath().split("/|\\\\");
                    if (ep[ep.length - 1].equals(p[p.length - 1]) && schemaHistory.getSuccess() == 1) {
                        return null;
                    }
                }
            }
            return new SQLScriptBo(props, model, f.getName(), f);
        }).filter(Objects::nonNull).sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    public void listSqlScriptBoCheck(List<SQLScriptBo> sqlScripts) {
        if (sqlScripts != null && sqlScripts.size() > 0) {
            Map<String, Long> versionGroup = sqlScripts.stream().collect(Collectors.groupingBy(b -> b.getModule() + ":" + b.getVersion(), Collectors.counting()));
            Optional<Map.Entry<String, Long>> error = versionGroup.entrySet().stream().filter(entry -> entry.getValue() > 1).findFirst();
            if (error.isPresent()) {
                String e = error.get().getKey();
                if (error.get().getKey().startsWith(COMMON + UNDER_LINE)) {
                    e = COMMON + e.substring(e.lastIndexOf(":"));
                }
                throw new IllegalArgumentException("数据脚本不符合规范：【 " + e + " 】下存在" + error.get().getValue() + "个版本相同的sql文件！");
            }
        }
    }

    public void increaseDbVersion(Connection conn, List<SQLScriptBo> sqlScripts, String tenantCol, String tenantVal) throws SQLException, IOException, NoSuchAlgorithmException, JSQLParserException {
        for (SQLScriptBo sqlScriptBo : sqlScripts) {
            this.increaseDbVersion(conn, sqlScriptBo, tenantCol, tenantVal);
        }
    }

    private void increaseDbVersion(Connection conn, SQLScriptBo sqlScriptBo, String tenantCol, String tenantVal) throws IOException, SQLException, NoSuchAlgorithmException, JSQLParserException {
        if (Files.getTemp() == null) {
            throw new IOException("临时文件获取异常！");
        }
        String checksum = Files.getFileChecksum(MessageDigest.getInstance("MD5"), sqlScriptBo.getFile());
        // 支持初始化脚本同一文件夹下多个脚本且没有版本号的情况
        SchemaHistoryEntity entity = schemaHistoryService.queryByVersion(conn, props, sqlScriptBo.getModule(), sqlScriptBo.getVersion());
        boolean insertFlag = true;
        if (entity == null) {
            entity = new SchemaHistoryEntity();
            entity.setCreatedBy("auto");
        } else if (entity.getSuccess() == 1) {
            return;
        } else if (entity.getChecksum().equals(checksum)) {
            return;
        } else {
            insertFlag = false;
            entity.setUpdateBy("auto");
        }
        entity.setType(TYPE_SQL);
        entity.setModel(sqlScriptBo.getModule());
        entity.setScriptPath(sqlScriptBo.getFile().getPath().replace(Files.getTemp().getAbsolutePath() + File.separator, ""));
        entity.setMajorVersion(sqlScriptBo.getMajorVersion());
        entity.setMinorVersion(sqlScriptBo.getMinorVersion());
        entity.setPatchVersion(sqlScriptBo.getPatchVersion());
        entity.setVersion(sqlScriptBo.getVersion());
        entity.setDescription(sqlScriptBo.getDescription());
        entity.setChecksum(checksum);
        log.debug("数据库[{}] 执行增量脚本:{}", conn.getMetaData().getURL(), sqlScriptBo.getFileName());
        // 开始时间
        LocalDateTime startTime = LocalDateTime.now();
        SuccessNum successNum = new SuccessNum();
        try {
            int startNum = 1;
            if (!insertFlag) {
                startNum = entity.getCodeBlockNum() + 1;
            }
            // 解决在jar包下执行sql中有中文乱码的问题
            FileInputStream fis = new FileInputStream(sqlScriptBo.getFile());
            InputStreamReader isr = new InputStreamReader(fis, StandardCharsets.UTF_8);
            scriptRunner.runScript(conn, isr, successNum, startNum, tenantCol, tenantVal);
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 成功 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            entity.setSuccess(1);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
        } catch (Exception e) {
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            entity.setSuccess(0);
            entity.setCodeBlockNum(successNum.value());
            entity.setExecutionTime(mills);
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 失败 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            log.error("", e);
            throw e;
        } finally {
            if (insertFlag) {
                this.schemaHistoryService.insertOne(conn, props, entity);
            } else {
                this.schemaHistoryService.updateOne(conn, props, entity);
            }
        }
    }
}
