package website.dachuan.migration.dao;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.dbutils.*;
import org.apache.commons.dbutils.handlers.BeanHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.mapping.DatabaseIdProvider;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * @author yqb22
 */
@Slf4j
public class SchemaHistoryDao {
    private final QueryRunner queryRunner;
    private final RowProcessor processor;
    private final String col = "id, model, major_version, minor_version, patch_version, version, description, type, script_path, checksum, success, code_block_num, execution_time, created_by, created_time, update_by, update_time";

    public SchemaHistoryDao() {
        this.queryRunner = new QueryRunner();
        BeanProcessor bean = new GenerousBeanProcessor();
        this.processor = new BasicRowProcessor(bean);
    }

    public List<SchemaHistoryEntity> queryLastVersion(Connection conn, String tableName) throws SQLException {
        StringBuilder queryLastVersionSql = new StringBuilder("select ").append(col);
        queryLastVersionSql.append(" from ").append(tableName);
        queryLastVersionSql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        List<SchemaHistoryEntity> resList = queryRunner.query(conn, queryLastVersionSql.toString(), new BeanListHandler<>(SchemaHistoryEntity.class, processor));
        setLog(queryLastVersionSql.toString(), new Object[]{}, resList);
        return resList;
    }

    public SchemaHistoryEntity queryLastVersion(Connection conn, String tableName, String model) throws SQLException {
        StringBuilder queryLastVersionSql = new StringBuilder("select ").append(col);
        queryLastVersionSql.append(" from ").append(tableName);
        queryLastVersionSql.append(" where model = ? and success = 1");
        queryLastVersionSql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        SchemaHistoryEntity rest = queryRunner.query(conn, queryLastVersionSql.toString(), new BeanHandler<>(SchemaHistoryEntity.class, processor), model);
        setLog(queryLastVersionSql.toString(), new Object[]{model}, rest);
        return rest;
    }

    public SchemaHistoryEntity queryByVersion(Connection conn, String tableName, String model, String version) throws SQLException {
        StringBuilder queryByVersionSql = new StringBuilder("select ").append(col);
        queryByVersionSql.append(" from ").append(tableName);
        queryByVersionSql.append(" where model = ? and version = ?");
        queryByVersionSql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        SchemaHistoryEntity rest = queryRunner.query(conn, queryByVersionSql.toString(), new BeanHandler<>(SchemaHistoryEntity.class, processor), model, version);
        setLog(queryByVersionSql.toString(), new Object[]{model, version}, rest);
        return rest;
    }

    public List<SchemaHistoryEntity> queryDeployInit(Connection conn, String model, String tableName) throws SQLException {
        StringBuilder queryDeployInitSql = new StringBuilder("select ").append(col);
        queryDeployInitSql.append(" from ").append(tableName);
        queryDeployInitSql.append(" where model = ?");
        List<SchemaHistoryEntity> rest = queryRunner.query(conn, queryDeployInitSql.toString(), new BeanListHandler<>(SchemaHistoryEntity.class, processor), model);
        setLog(queryDeployInitSql.toString(), new Object[]{model}, rest);
        return rest;
    }

    public SchemaHistoryEntity queryBaseLine(Connection conn, String tableName) throws SQLException {
        StringBuilder queryBaseLineSql = new StringBuilder("select ").append(col);
        queryBaseLineSql.append(" from ").append(tableName);
        queryBaseLineSql.append(" where type = 'BaseLine'");
        SchemaHistoryEntity rest = queryRunner.query(conn, queryBaseLineSql.toString(), new BeanHandler<>(SchemaHistoryEntity.class, processor));
        setLog(queryBaseLineSql.toString(), new Object[]{}, rest);
        return rest;
    }

    public void insertOne(Connection conn, String tableName, SchemaHistoryEntity entity) throws SQLException {
        boolean canAutoIncrement = DatabaseIdProvider.getDatabaseCanAutoIncrement(conn);
        StringBuilder insetSql = new StringBuilder();
        String colInsert = "model, major_version, minor_version, patch_version, version, description, type, script_path, checksum, success, code_block_num, execution_time, created_by";
        insetSql.append("insert into ").append(tableName).append(" ( ");
        if (!canAutoIncrement) {
            insetSql.append("id, ");
        }
        insetSql.append(colInsert).append(" ) ");
        insetSql.append(" values ");
        List<Object> params = new ArrayList<>();
        if (canAutoIncrement) {
            insetSql.append(" (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        } else {
            insetSql.append(" ( ");
            insetSql.append("seq_");
            insetSql.append(tableName);
            insetSql.append("_id.nextval,");
            insetSql.append("?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        }
        params.add(entity.getModel());
        params.add(entity.getMajorVersion());
        params.add(entity.getMinorVersion());
        params.add(entity.getPatchVersion());
        params.add(entity.getVersion());
        params.add(entity.getDescription());
        params.add(entity.getType());
        params.add(entity.getScriptPath());
        params.add(entity.getChecksum());
        params.add(entity.getSuccess());
        params.add(entity.getCodeBlockNum());
        params.add(entity.getExecutionTime());
        params.add(entity.getCreatedBy());
        int rs = queryRunner.execute(conn, insetSql.toString(), params.toArray());
        setLog(insetSql.toString(), params.toArray(), rs);
    }

    public int updateOne(Connection conn, String tableName, SchemaHistoryEntity entity) throws SQLException {
        if (entity.getId() == null) {
            throw new SQLException("更新程序，主键不能为空！");
        }
        List<Object> params = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        sb.append("update ").append(tableName).append(" set ");
        if (entity.getModel() != null && entity.getModel().length() != 0) {
            sb.append(" model = ?,");
            params.add(entity.getModel());
        }
        if (entity.getMajorVersion() != null) {
            sb.append(" major_version = ?,");
            params.add(entity.getMajorVersion());
        }
        if (entity.getMinorVersion() != null) {
            sb.append(" minor_version = ?,");
            params.add(entity.getMinorVersion());
        }
        if (entity.getPatchVersion() != null) {
            sb.append(" patch_version = ?,");
            params.add(entity.getPatchVersion());
        }
        if (entity.getVersion() != null && entity.getVersion().length() != 0) {
            sb.append(" version = ?,");
            params.add(entity.getVersion());
        }
        if (entity.getDescription() != null && entity.getDescription().length() != 0) {
            sb.append(" description = ?,");
            params.add(entity.getDescription());
        }
        if (entity.getType() != null && entity.getType().length() != 0) {
            sb.append(" type = ?,");
            params.add(entity.getType());
        }
        if (entity.getScriptPath() != null && entity.getScriptPath().length() != 0) {
            sb.append(" script_path = ?,");
            params.add(entity.getScriptPath());
        }
        if (entity.getChecksum() != null && entity.getChecksum().length() != 0) {
            sb.append(" checksum = ?,");
            params.add(entity.getChecksum());
        }
        if (entity.getSuccess() != null) {
            sb.append(" success = ?,");
            params.add(entity.getSuccess());
        }
        if (entity.getCodeBlockNum() != null) {
            sb.append(" code_block_num = ?,");
            params.add(entity.getCodeBlockNum());
        }
        if (entity.getExecutionTime() != null) {
            sb.append(" execution_time = ?,");
            params.add(entity.getExecutionTime());
        }
        if (entity.getUpdateBy() != null && entity.getUpdateBy().length() != 0) {
            sb.append(" update_by = ?,");
            params.add(entity.getUpdateBy());
        }
        if (entity.getUpdateTime() != null) {
            sb.append(" update_time = ?,");
            params.add(new Timestamp(entity.getUpdateTime().getTime()));
        } else {
            sb.append(" update_time = ?,");
            params.add(new Timestamp(System.currentTimeMillis()));
        }
        sb.deleteCharAt(sb.length() - 1);
        sb.append(" where id = ?");
        params.add(entity.getId());
        int r = queryRunner.update(conn, sb.toString(), params.toArray());
        setLog(sb.toString(), params.toArray(), r);
        return r;
    }

    private void setLog(String sql, Object[] params, Object rest) {
        log.debug("sql:{}", sql);
        log.debug("params:{}", Arrays.toString(params));
        log.debug("result:{}", rest);
    }
}
