package website.dachuan.migration;

import lombok.extern.slf4j.Slf4j;
import net.sf.jsqlparser.JSQLParserException;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.IDBOperation;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.service.task.*;
import website.dachuan.migration.utils.DatabaseIdProvider;
import website.dachuan.migration.utils.SqlScriptCommon;

import java.io.IOException;
import java.security.InvalidParameterException;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

@Slf4j
class MigrationProcess {
    private final MigrationProps props;
    private final Map<String, IDBOperation> dbOperations;
    private final ISqlScriptRunner scriptRunner;
    private final ISchemaHistoryService schemaHistoryService;
    /**
     * 数据库版本控制任务队列
     */
    private final BlockingQueue<MigrationTask> tasks = new LinkedBlockingQueue<>();

    public MigrationProcess(MigrationProps props, Map<String, IDBOperation> dbOperations, ISqlScriptRunner scriptRunner,
                            ISchemaHistoryService schemaHistoryService) {
        this.props = props;
        this.dbOperations = dbOperations;
        this.scriptRunner = scriptRunner;
        this.schemaHistoryService = schemaHistoryService;
    }

    public void exec(Connection conn, List<String> tenantIds) throws SQLException, JSQLParserException, IOException, NoSuchAlgorithmException {
        IDBOperation dbOperation = dbOperations.get(DatabaseIdProvider.getDatabaseId(conn));
        if (dbOperation == null) {
            throw new InvalidParameterException("DBOperation 实现类未发现！");
        }
        OperationMode operationMode = this.chargeOperationMode(conn);
        assert operationMode != null;
        this.assemblyTaskChain(operationMode, tenantIds);
        while (!this.tasks.isEmpty()) {
            this.tasks.poll().doTask(conn);
        }
    }

    private void assemblyTaskChain(OperationMode operationMode, List<String> tenantIds) {
        switch (operationMode) {
            case DEPLOY_INIT:
                tasks.add(new CreateHistoryTableTask(props, scriptRunner));
                tasks.add(new DeployInitTableTask(props, new SqlScriptCommon(props, schemaHistoryService, scriptRunner)));
                if (props.getTenantData().isEnabled() && tenantIds != null && !tenantIds.isEmpty()) {
                    for (String tenantId : tenantIds) {
                        tasks.add(new TenantDataInitScriptTask(props, new SqlScriptCommon(props, schemaHistoryService, scriptRunner), tenantId));
                    }
                }
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            case BASELINE_CREATE:
                tasks.add(new CreateHistoryTableTask(props, scriptRunner));
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            case BASELINE_INIT:
                tasks.add(new InsertBaselineTask(props, schemaHistoryService));
                break;
            default:
                break;
        }
        tasks.add(new IncreaseVersionTask(props, new SqlScriptCommon(props, schemaHistoryService, scriptRunner)));
        if (props.getTenantData().isEnabled() && tenantIds != null && !tenantIds.isEmpty()) {
            for (String tenantId : tenantIds) {
                tasks.add(new TenantDataScriptTask(props, new SqlScriptCommon(props, schemaHistoryService, scriptRunner), tenantId));
            }
        }
    }

    private OperationMode chargeOperationMode(Connection conn) throws SQLException {
        // 判断当前database是否非空
        List<String> tbNames = dbOperations.get(DatabaseIdProvider.getDatabaseId(conn)).listTables(conn);
        if (tbNames.size() == 0) {
            // 当前database为空，首次启动服务，导入全部数据库脚本，并创建数据库版本控制表，并生成数据库版本记录。
            return OperationMode.DEPLOY_INIT;
        }
        // 如果当前database非空，但还没有数据库版本控制表,创建数据库版本控制表，生成baseline记录；然后做增量的sql脚本执行。
        if (tbNames.stream().noneMatch(props.getSchemaHistoryTableName()::equals)) {
            return OperationMode.BASELINE_CREATE;
        }
        // 查询版本记录表是否存在数据
        List<SchemaHistoryEntity> historyEntity = schemaHistoryService.queryLastVersion(conn, props);
        if (historyEntity == null || historyEntity.size() == 0) {
            // 已经存在数据库版本控制表，但是没有版本记录信息，生成baseline记录；然后做增量的sql脚本执行。
            return OperationMode.BASELINE_INIT;
        } else {
            // 已经存在数据库版本控制表，并且存在执行记录，做增量的sql脚本执行。
            return OperationMode.DEPLOY_INCREASE;
        }
    }

    public enum OperationMode {
        // 项目首次部署，数据库没有任何表。
        // 该操作会生成数据库版本控制表，执行数据库初始化脚本，更新数据库版本控制表数据。
        DEPLOY_INIT,

        // 项目增量部署，之前已经导入业务表与数据库版本控制表。
        // 该操作根据已有的数据库版本控制表中的记录判断哪些脚本需要执行，然后执行脚本并插入新的数据库版本记录。
        DEPLOY_INCREASE,

        // 一个已经上线的项目初次使用数据库版本控制，之前已经导入业务表，但没有数据库版本控制表。
        // 该操作会创建数据库版本控制表，并写入一条版本基线记录，然后基于属性配置的基线版本确定哪些脚本需要执行。
        // 执行脚本后向数据库版本控制表插入新的版本记录。
        BASELINE_CREATE,

        // 一个已经上线的项目初次使用数据库版本控制，之前已经导入业务表并生成数据库版本控制表，但数据库版本控制表没有基线数据。
        // 该操作会写入一条版本基线记录，然后基于属性配置的基线版本确定哪些脚本需要执行。
        // 执行脚本后向数据库版本控制表插入新的版本记录。
        BASELINE_INIT
    }
}
