package website.dachuan.migration.utils;

import net.sf.jsqlparser.JSQLParserException;
import net.sf.jsqlparser.expression.Expression;
import net.sf.jsqlparser.expression.ExpressionVisitorAdapter;
import net.sf.jsqlparser.expression.Parenthesis;
import net.sf.jsqlparser.expression.StringValue;
import net.sf.jsqlparser.expression.operators.conditional.AndExpression;
import net.sf.jsqlparser.expression.operators.conditional.OrExpression;
import net.sf.jsqlparser.expression.operators.relational.*;
import net.sf.jsqlparser.parser.CCJSqlParserUtil;
import net.sf.jsqlparser.schema.Column;
import net.sf.jsqlparser.statement.Statement;
import net.sf.jsqlparser.statement.delete.Delete;
import net.sf.jsqlparser.statement.insert.Insert;
import net.sf.jsqlparser.statement.select.SubSelect;
import net.sf.jsqlparser.statement.update.Update;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

public class SqlParseUtil {

    public static String addTenant(String sql, String tenantCol, String tenantVal) throws JSQLParserException {
        Statement statement = CCJSqlParserUtil.parse(sql);
        if (statement instanceof Insert) {
            return addTenant2Insert(statement, tenantCol, tenantVal);
        } else if (statement instanceof Update) {
            return addTenant2Update(statement, tenantCol, tenantVal);
        } else if (statement instanceof Delete) {
            return addTenant2Delete(statement, tenantCol, tenantVal);
        } else {
            return sql;
        }
    }

    /**
     * 为sql语句增加租户号
     *
     * @param statement statement.toString()
     * @param tenantCol 租户字段名称
     * @param tenantVal 租户值
     * @return 处理后的sql
     * @throws JSQLParserException 转化异常
     */
    private static String addTenant2Insert(Statement statement, String tenantCol, String tenantVal) throws JSQLParserException {
        if (!(statement instanceof Insert)) {
            return statement.toString();
        }
        Insert insert = (Insert) statement;
        List<Column> columns = insert.getColumns();
        if (columns == null || columns.size() == 0) {
            throw new JSQLParserException("insert 语句不规范，缺失columns！");
        }
        ItemsList values = insert.getItemsList();
        if (values != null) {
            values.accept(new ItemsListVisitorAdapter() {
                @Override
                public void visit(SubSelect subSelect) {
                    throw new UnsupportedOperationException("不支持！");
                }

                @Override
                public void visit(NamedExpressionList namedExpressionList) {
                    throw new UnsupportedOperationException("不支持！");
                }

                @Override
                public void visit(ExpressionList expressionList) {
                    AtomicInteger index = new AtomicInteger(-1);
                    Optional<Column> c = columns.stream().filter(column -> {
                        //每比对一个元素，数值加1
                        index.getAndIncrement();
                        return tenantCol.equals(parseColumnName(column.getColumnName()));
                    }).findFirst();
                    List<Expression> expressions = expressionList.getExpressions();
                    if (c.isPresent()) {
                        Expression tenantExpression = expressions.get(index.get());
                        if (!(tenantExpression instanceof StringValue)) {
                            throw new IllegalArgumentException("租户号类型异常，租户号必须为String！");
                        }
                        String it = ((StringValue) tenantExpression).getValue();
                        if (tenantVal.equals(it)) {
                            return;
                        } else {
                            throw new IllegalArgumentException("sql中 包含租户号与租户信息，但当租户信息与需要添加的租户信息不同！");
                        }
                    } else {
                        columns.add(new Column(tenantCol));
                        expressionList.addExpressions(new StringValue(tenantVal));
                    }
                }

                @Override
                public void visit(MultiExpressionList multiExprList) {
                    throw new UnsupportedOperationException("不支持！");
                }
            });
        }
        return insert.toString();
    }

    /**
     * 为sql语句增加租户号
     *
     * @param statement Statement
     * @param tenantCol 租户字段名称
     * @param tenantVal 租户值
     * @return 处理后的sql
     * @throws JSQLParserException 转化异常
     */
    private static String addTenant2Update(Statement statement, String tenantCol, String tenantVal) throws JSQLParserException {
        if (!(statement instanceof Update)) {
            return statement.toString();
        }
        Update update = (Update) statement;
        Expression newExpression = parseWhere(update.getWhere(), tenantCol, tenantVal);
        update.setWhere(newExpression);
        return update.toString();
    }

    /**
     * 为sql语句增加租户号
     *
     * @param statement Statement
     * @param tenantCol 租户字段名称
     * @param tenantVal 租户值
     * @return 处理后的sql
     * @throws JSQLParserException 转化异常
     */
    private static String addTenant2Delete(Statement statement, String tenantCol, String tenantVal) throws JSQLParserException {
        if (!(statement instanceof Delete)) {
            return statement.toString();
        }
        Delete delete = (Delete) statement;
        Expression newExpression = parseWhere(delete.getWhere(), tenantCol, tenantVal);
        delete.setWhere(newExpression);
        return delete.toString();
    }

    private static Expression parseWhere(Expression currentExpression, String tenantCol, String tenantVal) {
        EqualsTo equalsTo = new EqualsTo();
        equalsTo.setLeftExpression(new Column(tenantCol));
        equalsTo.setRightExpression(new StringValue(tenantVal));
        Expression newExpression;
        // 检验更新语句中的where条件中是否存在 tenantCol = tenantVal 表达式
        AtomicBoolean exist = new AtomicBoolean(false);
        if (currentExpression != null) {
            currentExpression.accept(new ExpressionVisitorAdapter() {
                @Override
                public void visit(EqualsTo expr) {
                    Column column = (Column) expr.getLeftExpression();
                    if (tenantCol.equals(parseColumnName(column.getColumnName()))) {
                        if (!(expr.getRightExpression() instanceof StringValue)) {
                            throw new IllegalArgumentException("租户号类型异常，租户号必须为String！");
                        } else {
                            String it = ((StringValue) expr.getRightExpression()).getValue();
                            if (tenantVal.equals(it)) {
                                exist.set(true);
                            } else {
                                throw new IllegalArgumentException("sql中 包含租户号与租户信息，但当租户信息与需要添加的租户信息不同！");
                            }

                        }
                    }
                }
            });
        }
        if (exist.get()) {
            return currentExpression;
        }
        if (currentExpression == null) {
            newExpression = equalsTo;
        } else {
            if (currentExpression instanceof OrExpression) {
                newExpression = new AndExpression(new Parenthesis(currentExpression), equalsTo);
            } else {
                newExpression = new AndExpression(currentExpression, equalsTo);
            }
        }
        return newExpression;
    }

    private static String parseColumnName(String columnName) {
        if (columnName == null || columnName.isEmpty()) {
            return columnName;
        }
        return columnName.replaceAll("\"", "");
    }
}
