package website.dachuan.migration.service.task;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.bo.SuccessNum;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.utils.DatabaseIdProvider;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

@Slf4j
public class CreateHistoryTableTask implements MigrationTask {
    private final MigrationProps props;
    private final ISqlScriptRunner scriptRunner;

    public CreateHistoryTableTask(MigrationProps props, ISqlScriptRunner scriptRunner) {
        this.props = props;
        this.scriptRunner = scriptRunner;
    }

    @Override
    public Boolean doTask(Connection conn) throws SQLException {
        log.debug("数据库[{}] CreateHistoryTableTask doTask begin...", conn.getMetaData().getURL());
        String defaultCreateSchemaHistoryTableSqlPath = String.format("db/%s/schema_history.sql", DatabaseIdProvider.getDatabaseId(conn));
        InputStream is = this.getClass().getClassLoader().getResourceAsStream(defaultCreateSchemaHistoryTableSqlPath);
        assert is != null;
        Reader reader;
        if (this.props.getSchemaHistoryTableName() != null
                && this.props.getSchemaHistoryTableName().length() > 0
                && !"schema_history".equals(this.props.getSchemaHistoryTableName())) {
            reader = readSqlScript(is);
        } else {
            reader = new InputStreamReader(is);
        }
        scriptRunner.runScript(conn, reader, new SuccessNum());
        log.debug("数据库[{}] CreateHistoryTableTask doTask end...", conn.getMetaData().getURL());
        return true;
    }

    private Reader readSqlScript(InputStream is) {
        List<String> sqlScript = new ArrayList<>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8))) {
            // 生成sql构造器
            StringBuilder sqlBuilder = new StringBuilder();
            String line;
            while ((line = br.readLine()) != null) {
                // 去除首位空白字符
                line = line.trim();
                // 去除空行、注释行
                if (line.length() == 0 || line.startsWith("--")) {
                    continue;
                }
                if (line.contains("schema_history")) {
                    line = line.replace("schema_history", props.getSchemaHistoryTableName());
                }
                if (line.endsWith(";")) {
                    // 如果该行以";"结尾，则认为该条sql语句结束
                    sqlBuilder.append(line);
                    // 将sql构造器转为sql语句，加入sql语句集合
                    sqlScript.add(sqlBuilder.toString());
                    // 清空sql构造器
                    sqlBuilder.delete(0, sqlBuilder.length());
                } else {
                    // 如果该行没有以";"结尾，则认为该条sql语句尚未结束
                    sqlBuilder.append(line).append(" \n");
                }
            }
            // 特殊场景处理：如果sql脚本最后一条sql语句没有写";"结尾，则需要将非空的sql构造器转为sql语句并加入sql语句集合。
            if (sqlBuilder.length() > 0) {
                sqlBuilder.append(";");
                sqlScript.add(sqlBuilder.toString());
                sqlBuilder.delete(0, sqlBuilder.length());
            }
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
        return sqlScript.stream().reduce((a, b) -> a + "\n" + b).map(StringReader::new).orElse(null);
    }
}
