package website.dachuan.migration.service.task;

import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.MigrationException;
import website.dachuan.migration.bo.SQLScriptBo;
import website.dachuan.migration.entity.SchemaHistoryEntity;
import website.dachuan.migration.props.MigrationProps;
import website.dachuan.migration.service.ISchemaHistoryService;
import website.dachuan.migration.service.ISqlScriptRunner;
import website.dachuan.migration.service.MigrationTask;
import website.dachuan.migration.utils.Files;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.URL;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.SQLException;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
public class IncreaseVersionTask implements MigrationTask {
    private final MigrationProps props;
    private final ISchemaHistoryService schemaHistoryService;
    private final ISqlScriptRunner scriptRunner;

    public IncreaseVersionTask(MigrationProps props, ISchemaHistoryService schemaHistoryService, ISqlScriptRunner scriptRunner) {
        this.props = props;
        this.schemaHistoryService = schemaHistoryService;
        this.scriptRunner = scriptRunner;
    }

    @Override
    public Boolean doTask(Connection conn) throws IOException, SQLException, NoSuchAlgorithmException {
        log.debug("数据库[{}] IncreaseVersionTask doTask begin ...", conn.getMetaData().getURL());
        // 获取文件路径下的所有要执行的文件
        List<SQLScriptBo> scripts = this.createSqlScriptBo(conn);
        // 分步执行文件
        this.increaseDbVersion(conn, scripts);
        log.debug("数据库[{}] IncreaseVersionTask doTask end ...", conn.getMetaData().getURL());
        return true;
    }

    private List<SQLScriptBo> createSqlScriptBo(Connection conn) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts = new ArrayList<>();
        File folder;
        if (this.props.getScriptDirPath().startsWith("classpath:") || !this.props.getScriptDirPath().startsWith("/")) {
            URL url = this.getClass().getClassLoader().getResource(this.props.getScriptDirPath());
            assert url != null;
            folder = new File(url.getFile() + File.separator + conn.getMetaData().getDatabaseProductName());
        } else {
            folder = new File(this.props.getScriptDirPath() + File.separator + conn.getMetaData().getDatabaseProductName());
        }
        if (folder.exists()) {
            File[] l_model = folder.listFiles(File::isDirectory);
            File[] l_file = folder.listFiles(File::isFile);
            File[] l = folder.listFiles();
            assert l_file != null;
            assert l_model != null;
            assert l != null;
            if (l_file.length != l_model.length && l_file.length == l.length && l_file.length != 0) {
                // 全部是文件
                String model = "base";
                List<SQLScriptBo> temp = listSqlScriptBo(conn, model, folder);
                if (temp != null) {
                    sqlScripts.addAll(temp);
                }
            } else if (l_file.length != l_model.length && l_model.length == l.length && l_model.length != 0) {
                // 全部是文件夹（模块）
                for (File dir : l_model) {
                    List<SQLScriptBo> temp = listSqlScriptBo(conn, dir.getName(), dir);
                    if (temp != null && temp.size() > 0) {
                        sqlScripts.addAll(temp);
                    }
                }
            } else {
                throw new MigrationException("数据迁移失败！--迁移文件配置异常，请检查配置！");
            }

        } else {
            throw new MigrationException("数据迁移失败！--" + this.props.getScriptDirPath() + " is not Filesystem Directory!");
        }
        return sqlScripts;
    }

    private List<SQLScriptBo> listSqlScriptBo(Connection conn, String model, File dir) throws IOException, SQLException {
        List<SQLScriptBo> sqlScripts;
        List<File> sqlFiles;
        SchemaHistoryEntity entity = schemaHistoryService.queryLastVersion(conn, props, model);
        if (entity == null) {
            entity = schemaHistoryService.queryBaseLine(conn, props);
        }
        try (Stream<Path> paths = java.nio.file.Files.walk(Paths.get(dir.getAbsolutePath()))) {
            sqlFiles = paths.map(Path::toFile).filter(f -> f.isFile() && f.getPath().endsWith(props.getSqlMigrationSuffix()))
                    .collect(Collectors.toList());
        }
        if (sqlFiles.size() == 0) {
            return null;
        }
        int major = entity != null ? entity.getMajorVersion() : 0;
        int minor = entity != null ? entity.getMinorVersion() : 0;
        int patch = entity != null ? entity.getPatchVersion() : 0;
        sqlScripts = sqlFiles.stream().map(f -> new SQLScriptBo(props, model, f.getName(), f))
                .filter(bo -> bo.checkNeed(major, minor, patch))
                .sorted(SQLScriptBo::compareTo)
                .collect(Collectors.toList());
        return sqlScripts;
    }

    private void increaseDbVersion(Connection conn, List<SQLScriptBo> sqlScripts) throws SQLException, IOException, NoSuchAlgorithmException {
        for (SQLScriptBo sqlScriptBo : sqlScripts) {
            this.increaseDbVersion(conn, sqlScriptBo);
        }
    }

    private void increaseDbVersion(Connection conn, SQLScriptBo sqlScriptBo) throws IOException, SQLException, NoSuchAlgorithmException {
        String checksum = Files.getFileChecksum(MessageDigest.getInstance("MD5"), sqlScriptBo.getFile());
        SchemaHistoryEntity entity = schemaHistoryService.queryByVersion(conn, props, sqlScriptBo.getModule(), sqlScriptBo.getVersion());
        boolean flag = true;
        if (entity == null) {
            entity = new SchemaHistoryEntity();
            entity.setCreatedBy("auto");
        } else if (entity.getSuccess() == 1) {
            return;
        } else if (entity.getChecksum().equals(checksum)) {
            return;
        } else {
            flag = false;
            entity.setUpdateBy("auto");
            entity.setUpdateTime(new Date());
        }
        entity.setType("SQL");
        entity.setModel(sqlScriptBo.getModule());
        entity.setScriptPath(sqlScriptBo.getFile().getPath());
        entity.setMajorVersion(sqlScriptBo.getMajorVersion());
        entity.setMinorVersion(sqlScriptBo.getMinorVersion());
        entity.setPatchVersion(sqlScriptBo.getPatchVersion());
        entity.setVersion(sqlScriptBo.getVersion());
        entity.setDescription(sqlScriptBo.getDescription());
        entity.setChecksum(checksum);
        log.debug("数据库[{}] 执行增量脚本:{}", conn.getMetaData().getURL(), sqlScriptBo.getFileName());
        // 开始时间
        LocalDateTime startTime = LocalDateTime.now();
        try {
            scriptRunner.runScript(conn, new FileReader(sqlScriptBo.getFile()));
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 成功 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            entity.setSuccess(1);
            entity.setExecutionTime(mills);
        } catch (IOException | SQLException e) {
            LocalDateTime stopTime = LocalDateTime.now();
            long mills = Duration.between(startTime, stopTime).toMillis();
            entity.setSuccess(0);
            entity.setExecutionTime(mills);
            log.debug("数据库[{}] 执行增量脚本:{} , 状态: 失败 , 执行耗时 : {} ms.", conn.getMetaData().getURL(), sqlScriptBo.getFileName(), mills);
            log.error("", e);
            throw e;
        } finally {
            if (flag) {
                this.schemaHistoryService.insertOne(conn, props, entity);
            } else {
                this.schemaHistoryService.updateOne(conn, props, entity);
            }
        }
    }
}
