package website.dachuan.migration.dao;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.dbutils.*;
import org.apache.commons.dbutils.handlers.BeanHandler;
import org.apache.commons.dbutils.handlers.BeanListHandler;
import org.apache.commons.dbutils.handlers.ScalarHandler;
import website.dachuan.migration.entity.SchemaHistoryEntity;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

@Slf4j
public class SchemaHistoryDao {
    private final QueryRunner queryRunner;
    // 开启下划线->驼峰转换所用
    private final RowProcessor processor = new BasicRowProcessor(new GenerousBeanProcessor());
    private final String col = "id, model, major_version, minor_version, patch_version, version, description, type, script_path, checksum, success, execution_time, created_by, created_time, update_by, update_time";

    public SchemaHistoryDao() {
        this.queryRunner = new QueryRunner();
    }

    public List<SchemaHistoryEntity> queryLastVersion(Connection conn, String tableName) throws SQLException {
        StringBuilder query_last_version_sql = new StringBuilder("select ").append(col);
        query_last_version_sql.append(" from ").append(tableName);
        query_last_version_sql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        ResultSetHandler<List<SchemaHistoryEntity>> resultSetHandler = new BeanListHandler<>(SchemaHistoryEntity.class, processor);
        log.debug(query_last_version_sql.toString());
        return queryRunner.query(conn, query_last_version_sql.toString(), resultSetHandler);
    }

    public SchemaHistoryEntity queryLastVersion(Connection conn, String tableName, String model) throws SQLException {
        StringBuilder query_last_version_sql = new StringBuilder("select ").append(col);
        query_last_version_sql.append(" from ").append(tableName);
        query_last_version_sql.append(" where model = ? and success = 1");
        query_last_version_sql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        ResultSetHandler<SchemaHistoryEntity> resultSetHandler = new BeanHandler<>(SchemaHistoryEntity.class, processor);
        log.debug(query_last_version_sql.toString());
        log.debug("params[{}]", model);
        return queryRunner.query(conn, query_last_version_sql.toString(), resultSetHandler, model);
    }

    public SchemaHistoryEntity queryByVersion(Connection conn, String tableName, String model, String version) throws SQLException {
        StringBuilder query_by_version_sql = new StringBuilder("select ").append(col);
        query_by_version_sql.append(" from ").append(tableName);
        query_by_version_sql.append(" where model = ? and version = ?");
        query_by_version_sql.append(" order by major_version desc, minor_version desc, patch_version desc, created_time desc ");
        ResultSetHandler<SchemaHistoryEntity> resultSetHandler = new BeanHandler<>(SchemaHistoryEntity.class, processor);
        log.debug(query_by_version_sql.toString());
        log.debug("params[{},{}]", model, version);
        return queryRunner.query(conn, query_by_version_sql.toString(), resultSetHandler, model, version);
    }

    public SchemaHistoryEntity queryBaseLine(Connection conn, String tableName) throws SQLException {
        StringBuilder query_base_line_sql = new StringBuilder("select ").append(col);
        query_base_line_sql.append(" from ").append(tableName);
        query_base_line_sql.append(" where type = 'BaseLine';");
        ResultSetHandler<SchemaHistoryEntity> resultSetHandler = new BeanHandler<>(SchemaHistoryEntity.class, processor);
        log.debug(query_base_line_sql.toString());
        return queryRunner.query(conn, query_base_line_sql.toString(), resultSetHandler);
    }

    public SchemaHistoryEntity insertOne(Connection conn, String tableName, SchemaHistoryEntity entity) throws SQLException {
        StringBuilder inset_sql = new StringBuilder();
        String col_insert = "model, major_version, minor_version, patch_version, version, description, type, script_path, checksum, success, execution_time, created_by";
        inset_sql.append("insert into ").append(tableName).append(" ( ").append(col_insert).append(" ) ");
        inset_sql.append(" values ");
        inset_sql.append(" (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?);");
        log.debug(inset_sql.toString());
        log.debug("params[{},{},{},{},{},{},{},{},{},{},{},{}]", entity.getModel(), entity.getMajorVersion(), entity.getMinorVersion(), entity.getPatchVersion(),
                entity.getVersion(), entity.getDescription(), entity.getType(), entity.getScriptPath(),
                entity.getChecksum(), entity.getSuccess(), entity.getExecutionTime(), entity.getCreatedBy());
        Object rs = queryRunner.insert(conn, inset_sql.toString(), new ScalarHandler<>(),
                entity.getModel(), entity.getMajorVersion(), entity.getMinorVersion(), entity.getPatchVersion(),
                entity.getVersion(), entity.getDescription(), entity.getType(), entity.getScriptPath(),
                entity.getChecksum(), entity.getSuccess(), entity.getExecutionTime(), entity.getCreatedBy());
        entity.setId(Long.parseLong(rs.toString()));
        return entity;
    }

    public int updateOne(Connection conn, String tableName, SchemaHistoryEntity entity) throws SQLException {
        if (entity.getId() == null) {
            throw new SQLException("更新程序，主键不能为空！");
        }
        List<Object> params = new ArrayList<>();
        StringBuilder sb = new StringBuilder();
        sb.append("update ").append(tableName).append(" set ");
        if (entity.getModel() != null && entity.getModel().length() != 0) {
            sb.append(" model = ?,");
            params.add(entity.getModel());
        }
        if (entity.getMajorVersion() != null) {
            sb.append(" major_version = ?,");
            params.add(entity.getMajorVersion());
        }
        if (entity.getMinorVersion() != null) {
            sb.append(" minor_version = ?,");
            params.add(entity.getMinorVersion());
        }
        if (entity.getPatchVersion() != null) {
            sb.append(" patch_version = ?,");
            params.add(entity.getPatchVersion());
        }
        if (entity.getVersion() != null && entity.getVersion().length() != 0) {
            sb.append(" version = ?,");
            params.add(entity.getVersion());
        }
        if (entity.getDescription() != null && entity.getDescription().length() != 0) {
            sb.append(" description = ?,");
            params.add(entity.getDescription());
        }
        if (entity.getType() != null && entity.getType().length() != 0) {
            sb.append(" type = ?,");
            params.add(entity.getType());
        }
        if (entity.getScriptPath() != null && entity.getScriptPath().length() != 0) {
            sb.append(" script_path = ?,");
            params.add(entity.getScriptPath());
        }
        if (entity.getChecksum() != null && entity.getChecksum().length() != 0) {
            sb.append(" checksum = ?,");
            params.add(entity.getChecksum());
        }
        if (entity.getSuccess() != null) {
            sb.append(" success = ?,");
            params.add(entity.getSuccess());
        }
        if (entity.getExecutionTime() != null) {
            sb.append(" execution_time = ?,");
            params.add(entity.getExecutionTime());
        }
        if (entity.getUpdateBy() != null && entity.getUpdateBy().length() != 0) {
            sb.append(" update_by = ?,");
            params.add(entity.getUpdateBy());
        }
        if (entity.getUpdateTime() != null) {
            sb.append(" update_time = ?,");
            params.add(entity.getUpdateTime());
        }
        sb.deleteCharAt(sb.length() - 1);
        sb.append(" where id = ?");
        params.add(entity.getId());
        log.debug(sb.toString());
        log.debug("params:{}", params);
        return queryRunner.update(conn, sb.toString(), params.toArray());
    }
}
