package website.dachuan.migration.bo;

import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import website.dachuan.migration.props.MigrationProps;

import java.io.File;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * SQL脚本Entity
 *
 * @author zhaochun
 */
@Data
@Slf4j
public class SQLScriptBo implements Comparable<SQLScriptBo> {
    // sql脚本文件名正则表达式
    private Pattern PTN_SCRIPT_NAME;
    // 约定：当前数据目录下只能全部是文件夹（模块）或者是文件（迭代脚本）；模块下是迭代脚本
    private String module;
    // sql脚本文件名，格式为"[V]__[major].[minor].[patch]_[自定义名称].sql"
    private String fileName;
    // 文件
    private File file;
    // 主版本号，一个业务空间对应的主版本号，对应"x.y.z"中的x，只支持非负整数
    private int majorVersion;
    // 次版本号，一个业务空间对应的次版本号，对应"x.y.z"中的y，只支持非负整数
    private int minorVersion;
    // 补丁版本号，一个业务空间对应的补丁版本号，对应"x.y.z"中的z，只支持非负整数
    private int patchVersion;
    // 一个业务空间的完整版本号，格式为"[businessSpace]_V[majorVersion].[minorVersion].[patchVersion]"
    private String version;
    // 该sql脚本的自定义名称，支持大小写字母，数字与下划线
    private String description;

    /**
     * SQLScriptEntity构造方法
     *
     * @param props    配置参数
     * @param model    业务模块
     * @param fileName SQL脚本文件名
     * @param file     SQL脚本文件
     */
    public SQLScriptBo(MigrationProps props, String model, String fileName, File file) {
        PTN_SCRIPT_NAME = Pattern.compile(String.format("^%s%s(\\d+)\\.(\\d+)\\.(\\d+)%s(\\w+)%s$",
                props.getSqlMigrationPrefix(), props.getSqlMigrationSeparator(), props.getSqlMigrationSeparator(), props.getSqlMigrationSuffix()));
        this.module = model;
        this.fileName = fileName;
        this.file = file;
        Matcher matcher = PTN_SCRIPT_NAME.matcher(fileName);
        if (matcher.matches()) {
            this.majorVersion = Integer.parseInt(matcher.group(1));
            this.minorVersion = Integer.parseInt(matcher.group(2));
            this.patchVersion = Integer.parseInt(matcher.group(3));
            this.description = matcher.group(4);
            this.version = String.format("%s.%s.%s", majorVersion, minorVersion, patchVersion);
        } else {
            throw new RuntimeException(fileName + " format is not correct!");
        }
    }

    /**
     * 检查该SQL脚本是否需要执行
     *
     * @param major 数据库版本控制表最新记录的主版本号
     * @param minor 数据库版本控制表最新记录的次版本号
     * @param patch 数据库版本控制表最新记录的补丁版本号
     * @return 是否需要执行
     */
    public boolean checkNeed(int major, int minor, int patch) {
        if (this.majorVersion < major) {
            return false;
        } else if (this.majorVersion > major) {
            return true;
        }
        if (this.minorVersion < minor) {
            return false;
        } else if (this.minorVersion > minor) {
            return true;
        }
        return this.patchVersion > patch;
    }

    @Override
    public int compareTo(SQLScriptBo o) {
        if (this.getMajorVersion() == o.getMajorVersion()) {
            if (this.getMinorVersion() == o.getMinorVersion()) {
                return Integer.compare(this.getPatchVersion(), o.getPatchVersion());
            } else {
                return Integer.compare(this.getMinorVersion(), o.getMinorVersion());
            }
        } else {
            return Integer.compare(this.getMajorVersion(), o.getMajorVersion());
        }
    }
}
