/*
 * Decompiled with CFR 0.152.
 */
package voldemort.store.readonly.fetcher;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.NumberFormat;
import java.util.Arrays;
import java.util.Comparator;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.ObjectName;
import org.apache.commons.io.IOUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;
import voldemort.annotations.jmx.JmxGetter;
import voldemort.store.readonly.FileFetcher;
import voldemort.store.readonly.checksum.CheckSum;
import voldemort.store.readonly.fetcher.ConfigurableSocketFactory;
import voldemort.utils.ByteUtils;
import voldemort.utils.EventThrottler;
import voldemort.utils.JmxUtils;
import voldemort.utils.Props;
import voldemort.utils.Utils;

public class HdfsFetcher
implements FileFetcher {
    private static final Logger logger = Logger.getLogger(HdfsFetcher.class);
    private static final String DEFAULT_TEMP_DIR = new File(System.getProperty("java.io.tmpdir"), "hdfs-fetcher").getAbsolutePath();
    private static final int REPORTING_INTERVAL_BYTES = 0x6400000;
    private static final int DEFAULT_BUFFER_SIZE = 65536;
    private File tempDir;
    private final Long maxBytesPerSecond;
    private final int bufferSize;
    private final AtomicInteger copyCount = new AtomicInteger(0);

    public HdfsFetcher(Props props) {
        this(props.containsKey((Object)"fetcher.max.bytes.per.sec") ? Long.valueOf(props.getBytes("fetcher.max.bytes.per.sec")) : null, new File(props.getString("hdfs.fetcher.tmp.dir", DEFAULT_TEMP_DIR)), (int)props.getBytes("hdfs.fetcher.buffer.size", 65536L));
        logger.info((Object)("Created hdfs fetcher with temp dir = " + this.tempDir.getAbsolutePath() + " and throttle rate " + this.maxBytesPerSecond + " and buffer size " + this.bufferSize));
    }

    public HdfsFetcher() {
        this(null, null, 65536);
    }

    public HdfsFetcher(Long maxBytesPerSecond, File tempDir, int bufferSize) {
        this.tempDir = tempDir == null ? new File(DEFAULT_TEMP_DIR) : (File)Utils.notNull((Object)new File(tempDir, "hdfs-fetcher"));
        this.maxBytesPerSecond = maxBytesPerSecond;
        this.bufferSize = bufferSize;
        this.tempDir.mkdirs();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public File fetch(String fileUrl, String storeName) throws IOException {
        Path path = new Path(fileUrl);
        Configuration config = new Configuration();
        config.setInt("io.socket.receive.buffer", this.bufferSize);
        config.set("hadoop.rpc.socket.factory.class.ClientProtocol", ConfigurableSocketFactory.class.getName());
        FileSystem fs = path.getFileSystem(config);
        EventThrottler throttler = null;
        if (this.maxBytesPerSecond != null) {
            throttler = new EventThrottler(this.maxBytesPerSecond.longValue());
        }
        CopyStats stats = new CopyStats(fileUrl);
        ObjectName jmxName = JmxUtils.registerMbean((String)("hdfs-copy-" + this.copyCount.getAndIncrement()), (Object)stats);
        try {
            File storeDir = new File(this.tempDir, storeName + "_" + System.currentTimeMillis());
            storeDir.mkdir();
            File destination = new File(storeDir.getAbsoluteFile(), path.getName());
            boolean result = this.fetch(fs, path, destination, throttler, stats);
            if (result) {
                File file = destination;
                return file;
            }
            File file = null;
            return file;
        }
        finally {
            JmxUtils.unregisterMbean((ObjectName)jmxName);
        }
    }

    private boolean fetch(FileSystem fs, Path source, File dest, EventThrottler throttler, CopyStats stats) throws IOException {
        if (!fs.isFile(source)) {
            dest.mkdirs();
            FileStatus[] statuses = fs.listStatus(source);
            if (statuses != null) {
                Arrays.sort(statuses, new IndexFileLastComparator());
                byte[] origCheckSum = null;
                CheckSum.CheckSumType checkSumType = CheckSum.CheckSumType.NONE;
                CheckSum checkSumGenerator = null;
                CheckSum fileCheckSumGenerator = null;
                for (FileStatus status : statuses) {
                    if (status.getPath().getName().contains("checkSum.txt")) {
                        checkSumType = CheckSum.fromString(status.getPath().getName());
                        checkSumGenerator = CheckSum.getInstance(checkSumType);
                        fileCheckSumGenerator = CheckSum.getInstance(checkSumType);
                        FSDataInputStream input = fs.open(status.getPath());
                        origCheckSum = new byte[CheckSum.checkSumLength(checkSumType)];
                        input.read(origCheckSum);
                        input.close();
                        continue;
                    }
                    if (status.getPath().getName().startsWith(".")) continue;
                    File copyLocation = new File(dest, status.getPath().getName());
                    this.copyFileWithCheckSum(fs, status.getPath(), copyLocation, throttler, stats, fileCheckSumGenerator);
                    if (fileCheckSumGenerator == null || checkSumGenerator == null) continue;
                    checkSumGenerator.update(fileCheckSumGenerator.getCheckSum());
                }
                if (checkSumType != CheckSum.CheckSumType.NONE) {
                    byte[] newCheckSum = checkSumGenerator.getCheckSum();
                    return ByteUtils.compare((byte[])newCheckSum, origCheckSum) == 0;
                }
                return true;
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void copyFileWithCheckSum(FileSystem fs, Path source, File dest, EventThrottler throttler, CopyStats stats, CheckSum fileCheckSumGenerator) throws IOException {
        logger.info((Object)("Starting copy of " + source + " to " + dest));
        FSDataInputStream input = null;
        FileOutputStream output = null;
        try {
            int read;
            input = fs.open(source);
            output = new FileOutputStream(dest);
            byte[] buffer = new byte[this.bufferSize];
            while ((read = input.read(buffer)) >= 0) {
                if (read < this.bufferSize) {
                    buffer = ByteUtils.copy((byte[])buffer, (int)0, (int)read);
                }
                ((OutputStream)output).write(buffer);
                if (fileCheckSumGenerator != null) {
                    fileCheckSumGenerator.update(buffer);
                }
                if (throttler != null) {
                    throttler.maybeThrottle(read);
                }
                stats.recordBytes(read);
                if (stats.getBytesSinceLastReport() <= 0x6400000L) continue;
                NumberFormat format = NumberFormat.getNumberInstance();
                format.setMaximumFractionDigits(2);
                logger.info((Object)(stats.getTotalBytesCopied() / 0x100000L + " MB copied at " + format.format(stats.getBytesPerSecond() / 1048576.0) + " MB/sec"));
                stats.reset();
            }
            logger.info((Object)("Completed copy of " + source + " to " + dest));
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(output);
            IOUtils.closeQuietly((InputStream)input);
            throw throwable;
        }
        IOUtils.closeQuietly((OutputStream)output);
        IOUtils.closeQuietly((InputStream)input);
    }

    public static void main(String[] args) throws Exception {
        if (args.length != 1) {
            Utils.croak((String)("USAGE: java " + HdfsFetcher.class.getName() + " url storeName"));
        }
        String url = args[0];
        String storeName = args[1];
        long maxBytesPerSec = 0x40000000L;
        Path p = new Path(url);
        Configuration config = new Configuration();
        config.setInt("io.file.buffer.size", 65536);
        config.set("hadoop.rpc.socket.factory.class.ClientProtocol", ConfigurableSocketFactory.class.getName());
        config.setInt("io.socket.receive.buffer", 1038576);
        FileStatus status = p.getFileSystem(config).getFileStatus(p);
        long size = status.getLen();
        HdfsFetcher fetcher = new HdfsFetcher(maxBytesPerSec, null, 65536);
        long start = System.currentTimeMillis();
        File location = fetcher.fetch(url, storeName);
        double rate = (double)(size * 1000L) / (double)(System.currentTimeMillis() - start);
        NumberFormat nf = NumberFormat.getInstance();
        nf.setMaximumFractionDigits(2);
        System.out.println("Fetch to " + location + " completed: " + nf.format(rate / 1048576.0) + " MB/sec.");
    }

    /*
     * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
     */
    public static class IndexFileLastComparator
    implements Comparator<FileStatus> {
        @Override
        public int compare(FileStatus fs1, FileStatus fs2) {
            if (fs1.isDir()) {
                return fs2.isDir() ? 0 : -1;
            }
            if (fs1.getPath().getName().endsWith("checkSum.txt")) {
                return -1;
            }
            if (fs2.getPath().getName().endsWith("checkSum.txt")) {
                return 1;
            }
            if (fs1.getPath().getName().endsWith(".index")) {
                return fs2.getPath().getName().endsWith(".index") ? 0 : 1;
            }
            return 0;
        }
    }

    public static class CopyStats {
        private final String fileName;
        private volatile long bytesSinceLastReport;
        private volatile long totalBytesCopied;
        private volatile long lastReportNs;

        public CopyStats(String fileName) {
            this.fileName = fileName;
            this.totalBytesCopied = 0L;
            this.bytesSinceLastReport = 0L;
            this.lastReportNs = System.nanoTime();
        }

        public void recordBytes(long bytes) {
            this.totalBytesCopied += bytes;
            this.bytesSinceLastReport += bytes;
        }

        public void reset() {
            this.bytesSinceLastReport = 0L;
            this.lastReportNs = System.nanoTime();
        }

        public long getBytesSinceLastReport() {
            return this.bytesSinceLastReport;
        }

        @JmxGetter(name="totalBytesCopied", description="The total number of bytes copied so far in this transfer.")
        public long getTotalBytesCopied() {
            return this.totalBytesCopied;
        }

        @JmxGetter(name="bytesPerSecond", description="The rate of the transfer in bytes/second.")
        public double getBytesPerSecond() {
            double ellapsedSecs = (double)(System.nanoTime() - this.lastReportNs) / 1.0E9;
            return (double)this.bytesSinceLastReport / ellapsedSecs;
        }

        @JmxGetter(name="filename", description="The file path being copied.")
        public String getFilename() {
            return this.fileName;
        }
    }
}

