package com.alticast.mmuxclient;

import android.app.Instrumentation;
import android.app.UiModeManager;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.res.Configuration;
import android.graphics.Point;
import android.os.Bundle;
import android.os.Handler;
import android.os.IBinder;
import android.os.Message;
import android.os.Messenger;
import android.os.RemoteException;
import android.os.SystemClock;
import android.util.Log;
import android.view.Display;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.WindowManager;
import android.widget.Toast;

import com.alticast.voiceable.grpc.VoiceGateway;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

import static android.content.Context.UI_MODE_SERVICE;

/**
 * This class has everything a client application needs to connect to and utilize
 * functionalities of MMUX service.
 * <p>
 * Client app initially calls static function {@link #createASRContext(Context, String, Callback<Bundle>)}  to get
 * {@link ClientAPI.ASRContext} object.
 * Client can create more than one {@link ClientAPI.ASRContext} which are used at the same time when
 * speech is being recognized.  More recently created one has precedence when a speech corresponds
 * patterns defined by more than one {@link ClientAPI.ASRContext}.
 * <p>
 *
 * ClientAPI class instance is hidden from the client app.  One ClientAPI instance is
 * created per client app and shared among multiple {@link ClientAPI.ASRContext}s to communicate with
 * the MMUX service.
 * <p>
 * Client app is responsible for the life cycle management of {@link ClientAPI.ASRContext}.
 * When an Activity or Fragment is no more on screen and correspondent ASRContext is not relevant,
 * it should be destroyed immediately.
 *
 * @author yh.jeon@alticast.com
 *
 * Created by yh.jeon on 2017-03-09.
 */

public class ClientAPI {

    private static final String TAG = "ClientAPI";
    private static final boolean debug = true;

    // following MSG_ and PARAM_ constants are used for creating and parsing messages
    // between ClientAPI and the MMUX service

    public static final int MSG_CREATE_ASR_CONTEXT =  0;
    public static final int MSG_DESTROY_ASR_CONTEXT = 1;
    public static final int MSG_REGISTER_ASYNC_ERROR_HANDLER = 2;
    public static final int MSG_GET_LAST_RESPONSE = 100;
    public static final int MSG_ADD_ASR_PATTERN =   101;
    public static final int MSG_ADD_ENTITY_LIST =   102;
    public static final int MSG_SET_EXAMPLE_TEXTS = 103;
    public static final int MSG_SHOW_VOICE_PROMPT = 104;
    public static final int MSG_HIDE_VOICE_PROMPT = 105;
    public static final int MSG_CLEAR_ASR_PATTERNS = 106;
    public static final int MSG_ENABLE_SCREEN_CONTEXT = 107;
    public static final int MSG_DISABLE_SCREEN_CONTEXT = 108;
    public static final int MSG_HANDLE_API_EVENTS = 109;
    public static final int MSG_ADD_HANDLED_API_EVENTS = 110;

    public static final String PARAM_REQUEST_ID = "requestId";
    public static final String PARAM_CONTEXT_NAME = "contextName";
    public static final String PARAM_ENTITY_LIST = "entityList";
    public static final String PARAM_PATTERN = "pattern";
    public static final String PARAM_ERROR_MESSAGE = "errorMessage";
    public static final String PARAM_EXAMPLE_TEXTS = "exampleTexts";
    public static final String PARAM_RELAY_DATA = "relayData";
    public static final String PARAM_SUPPORTED_LANGUAGES = "supportedLanguages";
    public static final String PARAM_CURRENT_LANGUAGE = "currentLanguage";
    public static final String PARAM_ASR_ENGINE = "asrEngine";
    public static final String PARAM_PHRASE_SPOTTER = "phraseSpotter";

    public static final String PARAM_MATCH_CLASS = "matchClass";

    public static final String IMMEDIATE_MATCH_CLASS = "immediate";
    public static final String WAIT_END_MATCH_CLASS = "waitend";
    public static final String NORMAL_MATCH_CLASS = "normal";

    private static ClientAPI instance;
    private static Context context;
    private Messenger mmuxService = null;
    private ServiceConnection mmuxConnection;
    private Handler connectionHandler;
    private boolean bound;
    private List<ASRContext> asrContexts = new ArrayList<>();
    private Queue<Runnable> pendingTasks = new LinkedList<>();

    /**
     * Use this function to get {@link ClientAPI.ASRContext} instance.
     * Binding to the MMUX service is asynchronous.  Therefore, creation of the instance does not
     * guarantee that the MMUX service is running and successfully bound.
     * @param context Android context of the client application
     * @param contextName Name of the ASR context to be created. Should be unique within a client app.
     * @return
     */
    public static ASRContext createASRContext(Context context, String contextName, Callback<Bundle> onCreated) {

        return getInstance(context)._createASRContext(context.getPackageName(), contextName, onCreated);

    }

    /**
     * When an {@link ClientAPI.ASRContext} is not needed, the client app should call this function
     * with the name of the ASR context to destroy corresponding data structure in the MMUX service.
     * @param asrContext Name of the ASR context which was used for its creation.
     */
    public static void destroyASRContext(ASRContext asrContext) {
        Log.v(TAG,"destroyASRContext:"+asrContext.name);
        asrContext.destroyASRContext();
        if (instance!=null) {
            instance.asrContexts.remove(asrContext);
        }
    }

    private ASRContext _createASRContext(String packageName, String asrContextName, Callback<Bundle> onCreated) {

        // ASRContext full name is package_name:context_name

        String name=packageName + ":" + asrContextName;

        // check if there is a context of the same name
        for(ASRContext c: asrContexts)
            if (c.name.equals(name))
                return c;

        ASRContext asrContext = new ASRContext(packageName+":"+asrContextName, onCreated);
        asrContexts.add(asrContext);
        return asrContext;
    }

    private static ClientAPI getInstance(Context ctx) {
        if (instance==null) {
            instance = new ClientAPI(ctx);
        }
        return instance;
    }

    private ClientAPI(Context ctx) {
        this.context = ctx;
        bind(context);

    }

    public ServiceConnection makeServiceConnection(){
        return new ServiceConnection() {
            public void onServiceConnected(ComponentName className, IBinder service) {
                Log.d(TAG, "onServiceConnected");
                if(service == null){
                    Log.d(TAG, "service is null");
                    context.unbindService(instance.mmuxConnection);
                    bound = false;
                }
                else {
                    mmuxService = new Messenger(service);
                    bound = true;
                    // XXX not thread safe?
                    if (debug && pendingTasks.size() > 0)
                        Log.d(TAG, "sending pending " + pendingTasks.size() + " messages ");
                    Log.d(TAG, "asrContexts: " + asrContexts.isEmpty());
                    while (!pendingTasks.isEmpty()) {
                        pendingTasks.remove().run();
                    }
                    if (!asrContexts.isEmpty()) {
                        for (ASRContext asrContext : asrContexts) {
                            asrContext.initialize();
                        }

                    }

                }
            }
            public void onServiceDisconnected(ComponentName className) {
                mmuxService = null;
                Log.d(TAG, "onServiceDisconnected");
                bound=false;
                connectionHandler = new Handler(){
                    @Override
                    public void handleMessage(Message msg){
                        Log.d(TAG, "handling message. bound: " + bound + " msg: " + msg);
                        context.unbindService(mmuxConnection);
                        bind(context);
                        if(bound){
                            Toast.makeText(context, "Service Connection Restored", Toast.LENGTH_LONG).show();
                            connectionHandler.removeMessages(0);
                            connectionHandler = null;
                            return;
                        }
                        else{

                            connectionHandler.sendEmptyMessageDelayed(0, 3000);
                        }
                    }
                };
                Toast.makeText(context, "Service Connection Lost, Trying to Reconnect", Toast.LENGTH_LONG).show();
                connectionHandler.sendEmptyMessageDelayed(0, 0);
            }
        };
    }

    public void bind(Context ctx){
        Log.d(TAG, "Trying to bind again.");
        mmuxConnection = makeServiceConnection();
        Intent i = new Intent();
        i.setComponent(new ComponentName("com.alticast.mmux", "com.alticast.mmux.ClientService"));
        boolean result=ctx.bindService(i, mmuxConnection, Context.BIND_AUTO_CREATE | Context.BIND_IMPORTANT);

        if (!result) {
            Log.e(TAG, "Bind to mmux service failed");
        }
    }

    public static void unbind(Context ctx) {
        if (instance!=null) {
            // destroy any remaining contexts
            for(ASRContext c: instance.asrContexts)
                c.destroyASRContext();

            ctx.unbindService(instance.mmuxConnection);
            instance.bound = false;
            instance = null;
        }
    }

    /**
     * Implement this interface to define callback function to be called asynchronously
     * by the MMUX service, for example, as a result of speech recognition.  The {@link #callback(Object)}
     * function will run in the application's UI thread.
     *
     * @param <T>
     */
    public interface Callback<T> {
        public void callback(T t);
    }

    public static class Entity{
        private Object value;
        private String type;
        public Entity(String type, Object value){
            this.value = value;
            this.type = type;
        }
        public Object getValue(){ return value; }
        public String getType(){ return type; }
    }

    /**
     * This class describes the result of speech recognition.
     */
    public static class ASRResult {
        public static final String PARAM_RESPONSE_TEXT = "responseFullText";
        public static final String PARAM_SPOKEN_RESPONSE = "SpokenResponse";
        public static final String PARAM_MATCHED_PATTERN = "MatchedPattern";
        public static final String PARAM_CONFIDENCE = "ConfidenceScore";
        public static final String PARAM_ENTITIES = "Entities";


        private String spokenResponse;
        private String matchedPattern;
        private float confidenceScore;
        private ArrayList<Entity> entities;

        public String getSpokenResponse() {
            return spokenResponse;
        }

        public void setSpokenResponse(String spokenResponse) {
            this.spokenResponse = spokenResponse;
        }

        public String getMatchedPattern() {
            return matchedPattern;
        }

        public void setMatchedPattern(String matchedPattern) {
            this.matchedPattern = matchedPattern;
        }

        public float getConfidenceScore() {
            return confidenceScore;
        }

        public void setConfidenceScore(float confidenceScore) {
            this.confidenceScore = confidenceScore;
        }

        public ArrayList<Entity> getMatchedEntities() {
            return entities;
        }

        public void setEntities(String entities) {
            ArrayList<Entity> ent = new ArrayList<>();
            if(entities != null) {
                try {
                    JSONArray jsonArray = new JSONArray(entities);
                    for (int i = 0; i < jsonArray.length(); i++){
                        JSONObject topObj = jsonArray.getJSONObject(i);
                        String type = topObj.getString("name");
                        String value = topObj.getJSONObject("value").getJSONObject("value").getString(type);
                        Entity entity = new Entity(type, value);
                        ent.add(entity);
                    }
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
            this.entities = ent;
        }
    }

    static class ASRResultHandler extends Handler {
        private ClientAPI.Callback<ASRResult> callback;
        public ASRResultHandler(ClientAPI.Callback<ASRResult> callback) {
            this.callback = callback;
        }
        public void handleMessage(Message msg) {
            ASRResult r = new ASRResult();
            Bundle b = msg.getData();
            r.setSpokenResponse(b.getString(ASRResult.PARAM_SPOKEN_RESPONSE));
            r.setMatchedPattern(b.getString(ASRResult.PARAM_MATCHED_PATTERN));
            try {
                r.setConfidenceScore(Float.parseFloat(b.getString(ClientAPI.ASRResult.PARAM_CONFIDENCE)));
            } catch (Exception ex) {
                // ignore
            }
            r.setEntities(b.getString(ASRResult.PARAM_ENTITIES));
            callback.callback(r);
        }
    }

    public static class ClientEvent {
        public static final String PARAM_ACTION_TYPE = "actionType";
        public static final String SCROLL_DOWN = "SCROLL_DOWN";
        public static final String SCROLL_UP = "SCROLL_UP";
        public static final String SCROLL_LEFT = "SCROLL_LEFT";
        public static final String SCROLL_RIGHT = "SCROLL_RIGHT";
        public static final String ON_SHOW_PROMPT = "ON_SHOW_PROMPT";
        public static final String ON_HIDE_PROMPT = "ON_HIDE_PROMPT";

        private String action;

        private static final int TARGET_ANDROID_VANILLA = 0;
        private static final int TARGET_ANDROID_TV = 1;
        private int targetDevice = TARGET_ANDROID_VANILLA;

        public void setActionType(String actionType){
            action = actionType;
        }

        public void performAction(){
            new Thread(new Runnable(){
                @Override
                public void run() {
                    Log.i("ClientAPI", "running scroll down, action: " + action);
                    Instrumentation inst = new Instrumentation();
                    Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
                    Point size = new Point();
                    display.getSize(size);
                    int SCREEN_WIDTH = size.x, SCREEN_HEIGHT = size.y;
                    int SCROLL_DIST = 100;
                    int X1 = 0, X2 = 0, Y1 = 0, Y2 = 0;

                    long downTime = SystemClock.uptimeMillis();
                    long eventTime = SystemClock.uptimeMillis();

                    if (action.equals(SCROLL_DOWN)) {
                        X1 = SCREEN_WIDTH/2; X2 = SCREEN_WIDTH/2;
                        Y1 = SCREEN_HEIGHT/2 + SCROLL_DIST; Y2 = SCREEN_HEIGHT/2;
                    } else if (action.equals(SCROLL_UP)) {
                        X1 = SCREEN_WIDTH/2; X2 = SCREEN_WIDTH/2;
                        Y1 = SCREEN_HEIGHT/2 - SCROLL_DIST; Y2 = SCREEN_HEIGHT/2;
                    } else if (action.equals(SCROLL_LEFT)) {
                        X1 = SCREEN_WIDTH/2 - SCROLL_DIST; X2 = SCREEN_WIDTH/2;
                        Y1 = SCREEN_HEIGHT/2; Y2 = SCREEN_HEIGHT/2;
                    } else if (action.equals(SCROLL_RIGHT)) {
                        X1 = SCREEN_WIDTH/2 + SCROLL_DIST; X2 = SCREEN_WIDTH/2;
                        Y1 = SCREEN_HEIGHT/2; Y2 = SCREEN_HEIGHT/2;
                    }
                    MotionEvent event1 = MotionEvent.obtain(downTime, downTime, MotionEvent.ACTION_DOWN, X1, Y1, 0);
                    inst.sendPointerSync(event1);
                    event1 = MotionEvent.obtain(downTime, eventTime+30, MotionEvent.ACTION_MOVE, X2, Y2, 0);
                    inst.sendPointerSync(event1);
                    event1 = MotionEvent.obtain(downTime, eventTime+50, MotionEvent.ACTION_UP, X2, Y2, 0);
                    inst.sendPointerSync(event1);
                }
            }).start();
        }
    }
    static class ClientEventHandler extends Handler {
        private ClientAPI.Callback<ClientEvent> callback;
        public ClientEventHandler(ClientAPI.Callback<ClientEvent> callback) {
            this.callback = callback;
        }
        public void handleMessage(Message msg) {
            ClientEvent c = new ClientEvent();
            Bundle b = msg.getData();
            c.setActionType(b.getString(ClientEvent.PARAM_ACTION_TYPE));
            callback.callback(c);
        }
    }

    class AsyncErrorHandler extends Handler {

        public void handleMessage(Message msg) {
            Bundle b = msg.getData();
            if (b==null) {
                displayMessage("Empty async error message");
                return;
            }
            String message = b.getString(PARAM_ERROR_MESSAGE);
            displayMessage((message==null)?"Empty async error message":message );
        }

        void displayMessage(String msg) {
            Toast.makeText(context, msg, Toast.LENGTH_LONG).show();
        }
    }

    public class ASRContext {

        private String name;
        private Handler callback = null;
        protected ASRContext(String name, final Callback<Bundle> onCreated) {
            Log.d(TAG, "context Created: " + name);
            this.name = name;
            this.callback = new Handler() {
                public void handleMessage(Message msg) {
                    if (onCreated!=null)
                        onCreated.callback(msg.getData());
                }
            };
            initialize();
        }
        public void initialize(){
            Log.d(TAG, "sending initialize message");
            registerAsyncErrorHandler();
            createASRContext();
        }
        private void sendMessage(int msgType, Handler handler, Bundle data, VoiceGateway.OpenAPI.Builder builder) {

            final Message msg = Message.obtain(null, msgType);
            if (handler!=null)
                msg.replyTo = new Messenger(handler);
            if (data==null) {
                data = new Bundle();
            }
            if(builder != null) {
                data.putByteArray(PARAM_RELAY_DATA, builder.build().toByteArray());
            }
            msg.setData(data);
            if (bound) {
                try {
                    mmuxService.send(msg);
                } catch (RemoteException ex) {
                    Log.e(TAG, "Cannot send msg to mmux service: " + ex.getMessage());
                }
            } else {
                pendingTasks.add(new Runnable() {
                    public void run() {
                        try {
                            mmuxService.send(msg);
                        } catch (RemoteException ex) {
                            Log.e(TAG, "Cannot send msg to mmux service: " + ex.getMessage());
                        }
                    }
                });
            }

        }

        private void sendMessage(int msgType, Bundle data, VoiceGateway.OpenAPI.Builder builder) {
            sendMessage(msgType, null, data, builder);
        }

        private void sendMessage(int msgType, Handler handler, VoiceGateway.OpenAPI.Builder builder) {
            sendMessage(msgType, handler, null, builder);
        }

        private boolean checkIfBound() {

            return true;
        }
        protected void registerAsyncErrorHandler(){
            sendMessage(MSG_REGISTER_ASYNC_ERROR_HANDLER, new AsyncErrorHandler(), null);
        }
        protected void createASRContext(){
            VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder();
            builder.putParams(PARAM_CONTEXT_NAME, name);
            sendMessage(MSG_CREATE_ASR_CONTEXT, callback, builder);

        }
        protected void destroyASRContext() {
            VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder();
            builder.putParams(PARAM_CONTEXT_NAME, name);

            sendMessage(MSG_DESTROY_ASR_CONTEXT, null, null, builder);
        }

        public void addASRPattern(String pattern, Callback<ASRResult> callback) {
            addASRPattern(pattern, callback, NORMAL_MATCH_CLASS);
        }

        public void addASRPattern(String pattern, Callback<ASRResult> callback, String matchClass ) {
            if (checkIfBound()) {
                Bundle b = new Bundle();
                VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder()
                        .setMsg(MSG_ADD_ASR_PATTERN)
                        .putParams(PARAM_CONTEXT_NAME, name)
                        .putParams(PARAM_PATTERN, pattern)
                        .putParams(PARAM_MATCH_CLASS, matchClass);
                sendMessage(MSG_ADD_ASR_PATTERN, new ASRResultHandler(callback), b, builder);
            }
        }

        public void clearASRPatterns(Callback<ASRResult> callback) {
            if (checkIfBound()) {
                VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder()
                        .setMsg(MSG_CLEAR_ASR_PATTERNS);
                sendMessage(MSG_CLEAR_ASR_PATTERNS, new ASRResultHandler(callback), builder);
            }
        }

        @SuppressWarnings("unused")

        public void addEntityList(String name, String[] entityList) {
            if (checkIfBound()) {
                Bundle b=new Bundle();
                b.putStringArray(ClientAPI.PARAM_ENTITY_LIST, entityList);
                sendMessage(MSG_ADD_ENTITY_LIST, b, null);
            }
        }

        @SuppressWarnings("unused")

        public void setExampleTexts(String[] examples) {
            if (checkIfBound()) {
                Bundle b = new Bundle();
                b.putStringArray(ClientAPI.PARAM_EXAMPLE_TEXTS, examples);
                sendMessage(MSG_SET_EXAMPLE_TEXTS, b, null);
            }
        }

        public void showVoicePrompt() {
            if (checkIfBound()) {
                sendMessage(MSG_SHOW_VOICE_PROMPT, null, null, null);
            }
        }

        public void hideVoicePrompt() {
            if (checkIfBound()) {
                sendMessage(MSG_HIDE_VOICE_PROMPT, null, null, null);
            }
        }

        public void enableScreenContext() {
            if (checkIfBound()) {
                VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder()
                        .setMsg(MSG_ENABLE_SCREEN_CONTEXT);
                sendMessage(MSG_ENABLE_SCREEN_CONTEXT, null, null, builder);
            }
        }

        public void disableScreenContext() {
            if (checkIfBound()) {
                VoiceGateway.OpenAPI.Builder builder = VoiceGateway.OpenAPI.newBuilder()
                        .setMsg(MSG_DISABLE_SCREEN_CONTEXT);
                sendMessage(MSG_DISABLE_SCREEN_CONTEXT, null, null, builder);
            }
        }

        public void openEventPipe(){
            if(checkIfBound()){
                sendMessage(MSG_HANDLE_API_EVENTS, null, null, null);
            }
        }

        public void registerHandledEvents(String eventName, Callback<ClientEvent> callback){
            if(checkIfBound()){
                Bundle b = new Bundle();
                b.putString(ClientEvent.PARAM_ACTION_TYPE, eventName);
                sendMessage(MSG_ADD_HANDLED_API_EVENTS, new ClientEventHandler(callback), b, null);
            }
        }

        public void onShowVoicePrompt(Callback<ClientEvent> callback){
            Bundle b = new Bundle();
            b.putString(ClientEvent.PARAM_ACTION_TYPE, ClientEvent.ON_SHOW_PROMPT);
            sendMessage(MSG_ADD_HANDLED_API_EVENTS, new ClientEventHandler(callback), b, null);
        }
        public void onHideVoicePrompt(Callback<ClientEvent> callback){
            Bundle b = new Bundle();
            b.putString(ClientEvent.PARAM_ACTION_TYPE, ClientEvent.ON_HIDE_PROMPT);
            sendMessage(MSG_ADD_HANDLED_API_EVENTS, new ClientEventHandler(callback), b, null);
        }
    }
}
