(ns uswitch.riemann-java-heap
  (:require [clojure.string :as string]
            [riemann.client :as riemann-client])
  (:import [java.lang Runtime]))

(defn mapply
  [f & args]
  (apply f (apply concat (butlast args) (last args))))

(defn- make-client
  [client-opts]
  (let [client-fn (if (= :udp (:protocol client-opts))
                    riemann.client/udp-client
                    riemann.client/tcp-client)]
    (mapply client-fn client-opts)))

(defn heap-stats
  "Returns a map with heap size stats."
  []
  (let [runtime (Runtime/getRuntime)
        total (.totalMemory runtime)
        free (.freeMemory runtime)
        max (.maxMemory runtime)
        used (- total free)
        usage (double (/ used max))]
    (array-map
     :total total
     :free  free
     :max   max
     :used  used
     :usage usage)))

(defn description
  "Generates nicely formatted description of heap-stats, suitable as a
  Riemann event description."
  [heap-stats]
  (let [mb (* 1024 1024)]
    (string/join
     "\n"
     (for [[k v] heap-stats]
       (if (= :usage k)
         (format "usage: %.2f%%" v)
         (format "%s: %.0f MB" (name k) (double (/ v mb))))))))

(defn riemann-event
  "Generates a Riemann event based on heap-stats."
  [process-name ttl heap-stats]
  {:service (format "%s heap usage" process-name)
   :ttl (/ ttl 1000)
   :metric (:usage heap-stats)
   :state (cond
           (> (:usage heap-stats) 0.95) "critical"
           (> (:usage heap-stats) 0.8)  "warning"
           :else                        "ok")
   :description (description heap-stats)
   :tags ["heap"]})

(defn report-heap-stats
  "Reports heap-stats under the given process-name."
  [client process-name ttl ack? heap-stats]
  (try
    (riemann-client/send-event
     client
     (riemann-event process-name ttl heap-stats)
     ack?)
    (catch Throwable t
      ;; silently ignore
      )))

(defprotocol HeapObserver
  (stop [this]))

(defn start-observer
  "Starts a heap observer and reports events to
  riemann-host. process-name is used to construct the service name."
  [client-opts process-name interval ttl]
  (let [ack? (:ask client-opts)
        client (delay (make-client client-opts))
        run? (atom true)]
    (future
      (while @run?
        (report-heap-stats @client process-name ttl ack? (heap-stats))
        (Thread/sleep interval)))
    (reify HeapObserver
      (stop [this] (reset! run? false)))))
