package userkit.sdk.service;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.IBinder;
import android.os.Parcel;
import android.support.annotation.Nullable;
import android.util.Log;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import userkit.sdk.UserKit;
import userkit.sdk.Utils;
import userkit.sdk.model.Event;
import userkit.sdk.model.EventsPack;
import userkit.sdk.restful.UserKitService;

/**
 * Created by khangnt on 8/15/16.
 * Email: khang.neon.1997@gmail.com
 */
public class SenderService extends Service {
    public static final String EVENT_QUEUE_FILE_NAME = "event_queue";
    public static final String ACTION_ENQUEUE_EVENT = "ACTION_ENQUEUE_EVENT";

    private static final long LOOP_INTERVAL = 5*60*1000L; // 5 minutes
    private static final int MAX_EVENT_PER_TIME = 20;
    public static final String EVENT_DATA = "event_data";
    public static final String TAG = "SenderService";

    protected List<Event> eventQueue;
    protected UserKitService userKitService;
    protected Looper looperThread;

    @Override
    public void onCreate() {
        super.onCreate();
        OkHttpClient okHttpClient = new OkHttpClient.Builder()
                .connectTimeout(30, TimeUnit.SECONDS)
                .writeTimeout(30, TimeUnit.SECONDS)
                .readTimeout(30, TimeUnit.SECONDS)
                .build();
        userKitService = new Retrofit.Builder()
                .baseUrl("http://userkit-staging.ap-southeast-1.elasticbeanstalk.com/client/v1/")
                .client(okHttpClient)
                .addConverterFactory(GsonConverterFactory.create())
                .build()
                .create(UserKitService.class);

        looperThread = new Looper();

        // Read event queue from file
        try {
            FileInputStream fis = openFileInput(EVENT_QUEUE_FILE_NAME);
            byte[] array = new byte[(int) fis.getChannel().size()];
            //noinspection ResultOfMethodCallIgnored
            fis.read(array, 0, array.length);
            fis.close();

            Parcel parcel = Parcel.obtain();
            parcel.unmarshall(array, 0, array.length);
            parcel.setDataPosition(0);
            eventQueue = new ArrayList<>();
            parcel.readTypedList(eventQueue, Event.CREATOR);
            parcel.recycle();
        } catch (IOException e) {
            e.printStackTrace();
        }
        if (eventQueue == null)
            eventQueue = new ArrayList<>();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Log.i(TAG, "onStartCommand: Receive: " + intent.getAction());
        if (ACTION_ENQUEUE_EVENT.equalsIgnoreCase(intent.getAction())) {
            Event event = intent.getParcelableExtra(EVENT_DATA);
            eventQueue.add(event);
        }

        if (!looperThread.isRunning())
            looperThread.start();
        return START_NOT_STICKY;
    }

    protected void _5minutesLoop() {
        boolean shouldStop = true;
        if (Utils.isNetworkAvailable(this)) {
            if (eventQueue.size() > 0) {
                UserKit userKit = UserKit.getInstance();
                final int eventCount = Math.min(MAX_EVENT_PER_TIME, eventQueue.size());
                EventsPack eventsPack = new EventsPack(userKit.getUserProfile().getProfileId(),
                        eventQueue.subList(0, eventCount));
                userKitService.postEvents(eventsPack, userKit.getApiToken())
                        .enqueue(new Callback<Void>() {
                            @Override
                            public void onResponse(Call<Void> call, Response<Void> response) {
                                Log.i(TAG, "onResponse: send event success (" + eventCount + ")");
                                Log.i(TAG, "onResponse: " + response.code());
                                Log.i(TAG, "onResponse: " + response.message());
                                eventQueue.subList(0, eventCount).clear();
                            }

                            @Override
                            public void onFailure(Call<Void> call, Throwable t) {
                                Log.e(TAG, "onFailure: send event error (" + eventCount + ")", t);
                            }
                        });
                shouldStop = false;
            }
        }

        if (shouldStop)
            stopSelf();
    }





    @Override
    public void onDestroy() {
        super.onDestroy();
        looperThread.stop();
        // save the event list was not sent
        try {
            FileOutputStream fos = openFileOutput(EVENT_QUEUE_FILE_NAME, Context.MODE_PRIVATE);
            Parcel parcel = Parcel.obtain();
            parcel.writeTypedList(eventQueue);
            fos.write(parcel.marshall());
            fos.close();
            parcel.recycle();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    private class Looper implements Runnable {
        private Thread thread;

        boolean isRunning() {
            return thread != null && thread.isAlive();
        }

        void start() {
            thread = new Thread(this);
            thread.start();
        }

        void stop() {
            if (thread != null)
                thread.interrupt();
            thread = null;
        }

        @Override
        public void run() {
            do {
                try {
                    _5minutesLoop();
                    Thread.sleep(LOOP_INTERVAL);
                } catch (InterruptedException ignore) {
                    return;
                }
            } while (true);
        }
    }
}
