package userkit.sdk.model;

import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.Nullable;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.lang.reflect.Type;
import java.util.AbstractList;
import java.util.HashMap;
import java.util.Map;

import userkit.sdk.exception.PropertyValueException;

/**
 * Created by khangnt on 8/12/16.
 * Email: khang.neon.1997@gmail.com
 */
public class Event implements Parcelable {
    protected String name;
    protected Map<String, Object> properties;

    public Event(String eventName) {
        this.name = eventName;
        this.properties = new HashMap<>();
    }

    public String getName() {
        return name;
    }

    /**
     * Set event name.
     * @param name event name.
     */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * Add or replace a property value with type <tt>String</tt>.
     * @param name property name.
     * @param value property value or null.
     * @return instance of current event.
     */
    public Event putProperty(String name, @Nullable String value) {
        this.properties.put(name, value);
        return this;
    }

    /**
     * Add or replace a property value with type <tt>Number</tt>.
     * @param name property name
     * @param value property value or null.
     * @return instance of current event.
     */
    public Event putProperty(String name, @Nullable Number value) {
        this.properties.put(name, value);
        return this;
    }

    /**
     * Add or replace a property value with type <tt>Boolean</tt>.
     * @param name property name
     * @param value property value or null.
     * @return instance of current event.
     */
    public Event putProperty(String name, @Nullable Boolean value) {
        this.properties.put(name, value);
        return this;
    }

    public Event putProperty(String name, @Nullable Object... value)  {
        if (value != null)
            for (Object element : value)
                if (element != null
                        && !(element instanceof String)
                        && !(element instanceof Boolean)
                        && !(element instanceof Number))
                    throw new PropertyValueException(element.getClass());
        this.properties.put(name, value);
        return this;
    }

    public Event putProperty(String name, @Nullable AbstractList<Object> value)  {
        if (value != null)
            for (Object element : value)
                if (element != null
                        && !(element instanceof String)
                        && !(element instanceof Boolean)
                        && !(element instanceof Number))
                    throw new PropertyValueException(element.getClass());
        this.properties.put(name, value);
        return this;
    }

    protected Event(Parcel in) {
        name = in.readString();
        String mapAsJSON = in.readString();
        Type type = new TypeToken<Map<String, Object>>(){}.getType();
        properties = new Gson().fromJson(mapAsJSON, type);
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(name);
        // serialize HashMap<String, Object> to json by Gson
        String mapAsJSON = new Gson().toJson(properties);
        dest.writeString(mapAsJSON);
    }

    @SuppressWarnings("unused")
    public static final Parcelable.Creator<Event> CREATOR = new Parcelable.Creator<Event>() {
        @Override
        public Event createFromParcel(Parcel in) {
            return new Event(in);
        }

        @Override
        public Event[] newArray(int size) {
            return new Event[size];
        }
    };

}
