package userkit.sdk.profile;

import android.content.Context;
import android.content.SharedPreferences;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Date;
import java.util.List;

import okhttp3.RequestBody;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import userkit.sdk.UserKit;
import userkit.sdk.Utils;
import userkit.sdk.api.ErrorResponse;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.callback.DefaultCallback;
import userkit.sdk.callback.ICallback;
import userkit.sdk.data.ArraySearchData;
import userkit.sdk.data.ProfileIdAliasData;
import userkit.sdk.exception.ServerResponseErrorException;
import userkit.sdk.exception.UserProfileNullException;

/**
 * Created by khangnt on 8/15/16.
 * Email: khang.neon.1997@gmail.com
 */
public class UserProfile {
    public static final String USER_PROFILE_PREFERENCES = "userkit_user_profile";
    public static final String PROFILE_ID = "profileId";
    private Context context;
    private String profileId;

    private static UserProfile singleInstance;

    @Nullable
    public synchronized static UserProfile getCurrentProfile() {
        if (singleInstance == null) {
            Context context = UserKit.getInstance().getContext();
            SharedPreferences preferences = context.getSharedPreferences(USER_PROFILE_PREFERENCES, Context.MODE_PRIVATE);
            String profileId = preferences.getString(PROFILE_ID, null);
            if (profileId != null) {
                singleInstance = new UserProfile(context, profileId);
            }
        }
        return singleInstance;
    }

    public static void createNewProfile(@NonNull String identifyId, final ICallback<UserProfile> callback) {
        Context context = UserKit.getInstance().getContext();
        postCreateProfile(new UserProfile(context, identifyId), callback);
    }

    public static void createNewProfile(final ICallback<UserProfile> callback) {
        Context context = UserKit.getInstance().getContext();
        postCreateProfile(new UserProfile(context, null), callback);
    }


    private UserProfile(Context context, @Nullable String identifyId) {
        this.context = context;
        if (identifyId != null)
            this.profileId = identifyId;
        else
            initDefaultValue();
    }

    private void initDefaultValue() {
        String android_id = Settings.Secure.getString(context.getContentResolver(),
                Settings.Secure.ANDROID_ID);
        this.profileId = Utils.md5("android_" + android_id + "_" + new Date().getTime());
    }

    protected synchronized void saveUserProfile() {
        SharedPreferences preferences = context.getSharedPreferences(USER_PROFILE_PREFERENCES, Context.MODE_PRIVATE);
        preferences.edit()
                .putString(PROFILE_ID, profileId)
                .apply();
    }

    public void identify(@NonNull String id) {
        this.profileId = id;
        saveUserProfile();
    }

    public String getIdentifiedId() {
        return profileId;
    }

    public void mapProfileId(String newProfileId, final DefaultCallback callback) {
        UserKitService userKitService = ServiceGenerator.buildUserKitService();
        Call<Void> call = userKitService.mapProfileId(new ProfileIdAliasData(this.profileId, newProfileId));
        call.enqueue(new Callback<Void>() {
            @Override
            public void onResponse(Call<Void> call, Response<Void> response) {
                if (response.isSuccessful()) {
                    callback.onSuccess();
                } else {
                    ErrorResponse errorResponse = new ErrorResponse(response);
                    callback.onError(new ServerResponseErrorException(errorResponse));
                }
            }

            @Override
            public void onFailure(Call<Void> call, Throwable t) {
                callback.onError(t);
            }
        });
    }

    public void getProperties(String keys, final ICallback<JSONObject> callback) {
        UserKitService userKitService = ServiceGenerator.buildUserKitService();
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        userKitService.getProperties(profileId, keys).enqueue(new Callback<String>() {
            @Override
            public void onResponse(Call<String> call, Response<String> response) {
                if (response.isSuccessful()) {
                    try {
                        callback.onSuccess(new JSONObject(response.body()));
                    } catch (Throwable t) {
                        onFailure(call, t);
                    }
                } else {
                    ErrorResponse errorResponse = new ErrorResponse(response);
                    callback.onError(new ServerResponseErrorException(errorResponse));
                }
            }

            @Override
            public void onFailure(Call<String> call, Throwable t) {
                callback.onError(t);
            }
        });
    }

    public void getProperties(final ICallback<JSONObject> callback, String... keyArray) {
        if (keyArray == null || keyArray.length == 0)
            throw new IllegalArgumentException("Key array must not be empty or null.");
        String keys = "";
        for (int i = 0; i < keyArray.length - 1; i++) {
            keys += keyArray[i] + ",";
        }
        keys += keyArray[keyArray.length - 1];
        getProperties(keys, callback);
    }

    public void getProperties(List<String> keyArray, final ICallback<JSONObject> callback) {
        if (keyArray == null || keyArray.size() == 0)
            throw new IllegalArgumentException("Key array must not be empty or null.");
        String keys = "";
        for (int i = 0; i < keyArray.size() - 1; i++) {
            keys += keyArray.get(i) + ",";
        }
        keys += keyArray.get(keyArray.size() - 1);
        getProperties(keys, callback);
    }


    /**
     * @param key         query key
     * @param queryObject Must be: String, Boolean, Number, JSONObject or JSONArray.
     * @param callback    callback
     * @throws JSONException if queryObject is invalid.
     */
    public void searchInsideArray(String key, Object queryObject, final ICallback<JSONArray> callback) throws JSONException {
        UserKitService userKitService = ServiceGenerator.buildUserKitService();
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        RequestBody body = Utils.buildRequestBody(new ArraySearchData(profileId, key, queryObject));
        userKitService.searchInsideArray(body)
                .enqueue(new Callback<String>() {
                    @Override
                    public void onResponse(Call<String> call, Response<String> response) {
                        if (response.isSuccessful()) {
                            try {
                                callback.onSuccess(new JSONArray(response.body()));
                            } catch (JSONException e) {
                                onFailure(call, e);
                            }
                        } else {
                            onFailure(call, new ServerResponseErrorException(new ErrorResponse(response)));
                        }
                    }

                    @Override
                    public void onFailure(Call<String> call, Throwable t) {
                        callback.onError(t);
                    }
                });
    }

    /**
     * @param key         query key
     * @param queryObject Must be: String, Boolean, Number, JSONObject or JSONArray.
     * @param callback    callback
     * @throws JSONException if queryObject is invalid.
     */
    public void removeElementsInsideArray(String key, Object queryObject, final DefaultCallback callback) throws JSONException {
        UserKitService userKitService = ServiceGenerator.buildUserKitService();
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        RequestBody body = Utils.buildRequestBody(new ArraySearchData(profileId, key, queryObject));
        userKitService.removeElementsInsideArray(body)
                .enqueue(new Callback<Void>() {
                    @Override
                    public void onResponse(Call<Void> call, Response<Void> response) {
                        if (response.isSuccessful())
                            callback.onSuccess();
                        else
                            callback.onError(new ServerResponseErrorException(new ErrorResponse(response)));
                    }

                    @Override
                    public void onFailure(Call<Void> call, Throwable t) {
                        callback.onError(t);
                    }
                });
    }

    private static void postCreateProfile(final UserProfile newProfile, final ICallback<UserProfile> callback) {
        try {
            newProfile.edit()
                    .set(new JSONObject()
                            .put("platform", "android")
                            .put("country_code", "vn"))
                    .commit(new DefaultCallback() {
                        @Override
                        public void onSuccess() {
                            singleInstance = newProfile;
                            singleInstance.saveUserProfile();       // Update current profile
                            callback.onSuccess(newProfile);
                        }

                        @Override
                        public void onError(Throwable t) {
                            callback.onError(t);
                        }
                    });
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public ProfileEditor edit() {
        return new ProfileEditor(getIdentifiedId());
    }

}
