package userkit.sdk.service;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.IBinder;
import android.os.Parcel;
import android.support.annotation.Nullable;
import android.util.Log;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;
import userkit.sdk.Logging;
import userkit.sdk.Utils;
import userkit.sdk.api.ErrorResponse;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.data.Event;
import userkit.sdk.data.EventsData;
import userkit.sdk.exception.UserProfileNullException;
import userkit.sdk.profile.UserProfile;

/**
 * Created by khangnt on 8/15/16.
 * Email: khang.neon.1997@gmail.com
 */
public class AnalyticService extends Service {
    public static final String TAG = "AnalyticService";

    public static final String PENDING_TASKS = "userkit_pending_tasks";

    public static final String ACTION_ENQUEUE_EVENT = "ACTION_ENQUEUE_EVENT";

    public static final String EVENT_DATA = "event_data";
    public static final int MAX_EVENT_PER_TIME = 20;
    public static final int SLEEP_TIME_AFTER_ERROR = 3000;

    protected UserKitService userKitService;
    protected WorkerThread workerThread;
    protected List<Event> eventQueue;

    @Override
    public void onCreate() {
        super.onCreate();
        userKitService = ServiceGenerator.buildUserKitService();

        workerThread = new WorkerThread();

        eventQueue = new ArrayList<>();
        // Read pending tasks from file
        try {
            FileInputStream fis = openFileInput(PENDING_TASKS);
            byte[] array = new byte[(int) fis.getChannel().size()];
            //noinspection ResultOfMethodCallIgnored
            fis.read(array, 0, array.length);
            fis.close();

            Parcel parcel = Parcel.obtain();
            parcel.unmarshall(array, 0, array.length);
            parcel.setDataPosition(0);

            parcel.readTypedList(eventQueue, Event.CREATOR);

            parcel.recycle();
        } catch (IOException ignore) {
        }
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Log.i(TAG, "onStartCommand: Receive: " + intent.getAction());
        if (ACTION_ENQUEUE_EVENT.equalsIgnoreCase(intent.getAction())) {
            Event event = intent.getParcelableExtra(EVENT_DATA);
            eventQueue.add(event);
        }
        saveEventQueue();
        if (!workerThread.isRunning())
            workerThread.start();
        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        workerThread.stop();
        saveEventQueue();
    }

    private void saveEventQueue() {
        try {
            FileOutputStream fos = openFileOutput(PENDING_TASKS, Context.MODE_PRIVATE);
            Parcel parcel = Parcel.obtain();
            parcel.writeTypedList(eventQueue);
            fos.write(parcel.marshall());
            fos.close();
            parcel.recycle();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    private class WorkerThread implements Runnable {
        private Thread thread;
        private static final long LOOP_INTERVAL = 60 * 1000L; // 1 minute
        private static final long MAX_FREE_TIME = 5 * 60 * 1000L; // 5 minutes

        boolean isRunning() {
            return thread != null && thread.isAlive();
        }

        void start() {
            thread = new Thread(this);
            thread.start();
        }

        void stop() {
            if (thread != null)
                thread.interrupt();
            thread = null;
        }

        @Override
        public void run() {
            int freeTime = 0;
            do {
                if (freeTime > MAX_FREE_TIME)
                    break;
                if (eventQueue.size() > 0 && Utils.isNetworkAvailable(AnalyticService.this)) {
                    freeTime = 0;
                    int eventCount = Math.min(eventQueue.size(), MAX_EVENT_PER_TIME);
                    List<Event> eventsToSend = eventQueue.subList(0, eventCount);
                    try {
                        UserProfile userProfile = UserProfile.getCurrentProfile();
                        if (userProfile == null)
                            throw new UserProfileNullException();
                        EventsData eventsData = new EventsData(userProfile.getIdentifiedId(), eventsToSend);
                        Response<Void> response = userKitService.postEvents(eventsData).execute();
                        if (response.isSuccessful()) {
                            Logging.d("WorkerThread.run", "Send events success, count: " + eventsToSend.size());
                            eventsToSend.clear();
                            saveEventQueue();
                        } else {
                            Logging.e("WorkerThread.run", new ErrorResponse(response));
                            Logging.d("WorkerThread.run", "Sleep " + SLEEP_TIME_AFTER_ERROR);
                            Thread.sleep(SLEEP_TIME_AFTER_ERROR);
                        }
                    } catch (Throwable throwable) {
                        throwable.printStackTrace();
                        break;
                    }
                } else {
                    freeTime += LOOP_INTERVAL;
                    try {
                        Thread.sleep(LOOP_INTERVAL);
                    } catch (InterruptedException ignore) {
                        return;
                    }
                }
            } while (true);
            stopSelf();
        }
    }
}
