package userkit.sdk.service;

import android.app.Service;
import android.content.Context;
import android.content.Intent;
import android.os.IBinder;
import android.os.Parcel;
import android.support.annotation.Nullable;
import android.util.Log;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import userkit.sdk.Constants;
import userkit.sdk.UserKit;
import userkit.sdk.Utils;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.data.Event;
import userkit.sdk.job.JobBase;
import userkit.sdk.job.JobQueue;
import userkit.sdk.job.PostAliasIdJob;
import userkit.sdk.job.PostEventJob;
import userkit.sdk.job.PostProfileJob;
import userkit.sdk.profile.ProfilePropertyEditorBase;

/**
 * Created by khangnt on 8/15/16.
 * Email: khang.neon.1997@gmail.com
 */
public class SenderService extends Service {
    public static final String TAG = "SenderService";

    public static final String PENDING_TASKS = "userkit_pending_tasks";

    public static final String ACTION_ENQUEUE_EVENT = "ACTION_ENQUEUE_EVENT";
    public static final String ACTION_PROFILE_PROPERTY_EDITED = "ACTION_PROFILE_PROPERTY_EDITED";
    public static final String ACTION_ALIAS_PROFILE_ID = "ACTION_ALIAS_PROFILE_ID";

    public static final String EVENT_DATA = "event_data";
    public static final String PROFILE_PROPERTY_EDITOR_DATA = "profile_property_editor_data";
    public static final String ALIAS_ID = "alias_id_data";

    protected UserKitService userKitService;
    protected WorkerThread workerThread;
    protected JobQueue jobQueue;

    @Override
    public void onCreate() {
        super.onCreate();
        userKitService = ServiceGenerator.retrofit(Constants.USER_KIT_CLIENT)
                .create(UserKitService.class);

        workerThread = new WorkerThread();

        // Read pending tasks from file
        try {
            FileInputStream fis = openFileInput(PENDING_TASKS);
            byte[] array = new byte[(int) fis.getChannel().size()];
            //noinspection ResultOfMethodCallIgnored
            fis.read(array, 0, array.length);
            fis.close();

            Parcel parcel = Parcel.obtain();
            parcel.unmarshall(array, 0, array.length);
            parcel.setDataPosition(0);

            jobQueue = parcel.readParcelable(JobBase.class.getClassLoader());

            parcel.recycle();
        } catch (IOException e) {
            e.printStackTrace();
        }
        if (jobQueue == null)
            jobQueue = new JobQueue();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        Log.i(TAG, "onStartCommand: Receive: " + intent.getAction());
        if (ACTION_ENQUEUE_EVENT.equalsIgnoreCase(intent.getAction())) {
            Event event = intent.getParcelableExtra(EVENT_DATA);
            jobQueue.addJob(new PostEventJob(event));
        } else if (ACTION_PROFILE_PROPERTY_EDITED.equalsIgnoreCase(intent.getAction())) {
            ProfilePropertyEditorBase var1 = intent.getParcelableExtra(PROFILE_PROPERTY_EDITOR_DATA);
            jobQueue.addJob(new PostProfileJob(var1));
        } else if (ACTION_ALIAS_PROFILE_ID.equalsIgnoreCase(intent.getAction())) {
            String aliasId = intent.getStringExtra(ALIAS_ID);
            jobQueue.addJob(new PostAliasIdJob(aliasId));
        }
        saveJobQueue();
        if (!workerThread.isRunning())
            workerThread.start();
        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        workerThread.stop();
        saveJobQueue();
    }

    private void saveJobQueue() {
        try {
            FileOutputStream fos = openFileOutput(PENDING_TASKS, Context.MODE_PRIVATE);
            Parcel parcel = Parcel.obtain();
            parcel.writeParcelable(jobQueue, 0);
            fos.write(parcel.marshall());
            fos.close();
            parcel.recycle();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    private class WorkerThread implements Runnable {
        private Thread thread;
        private static final long LOOP_INTERVAL = 60 * 1000L; // 1 minute
        private static final long SHOULD_STOP = 5 * 60 * 1000L; // 5 minutes

        boolean isRunning() {
            return thread != null && thread.isAlive();
        }

        void start() {
            thread = new Thread(this);
            thread.start();
        }

        void stop() {
            if (thread != null)
                thread.interrupt();
            thread = null;
        }

        @Override
        public void run() {
            int freeTime = 0;
            do {
                if (freeTime > SHOULD_STOP)
                    break;
                if (jobQueue.getJobCount() > 0 && Utils.isNetworkAvailable(SenderService.this)) {
                    freeTime = 0;
                    JobBase job = jobQueue.getJob(0);
                    try {
                        job.run();
                        jobQueue.removeTopQueue();
                        saveJobQueue();
                    } catch (Throwable throwable) {
                        if (UserKit.DEBUG)
                            throw new RuntimeException(throwable);
                        else {
                            Log.e(TAG, "Server response error, stop sender service.", throwable);
                            break;
                        }
                    }
                } else {
                    freeTime += LOOP_INTERVAL;
                    try {
                        Thread.sleep(LOOP_INTERVAL);
                    } catch (InterruptedException ignore) {
                        return;
                    }
                }
            } while (true);
            stopSelf();
        }
    }
}
