package userkit.sdk.profile;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.AbstractList;
import java.util.Date;

import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import userkit.sdk.Constants;
import userkit.sdk.Properties;
import userkit.sdk.UserKit;
import userkit.sdk.Utils;
import userkit.sdk.api.ErrorHandler;
import userkit.sdk.api.ErrorResponse;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.callback.GetPropertiesCallback;
import userkit.sdk.exception.ServerResponseCodeException;
import userkit.sdk.exception.UserProfileNullException;
import userkit.sdk.service.SenderService;

/**
 * Created by khangnt on 8/15/16.
 * Email: khang.neon.1997@gmail.com
 */
public class UserProfile {
    public static final String USER_PROFILE_PREFERENCES = "userkit_user_profile";
    public static final String PROFILE_ID = "profileId";
    private Context context;
    private String profileId;

    private static UserProfile singleInstance;

    @Nullable
    public synchronized static UserProfile getCurrentProfile() {
        if (singleInstance == null) {
            Context context = UserKit.getInstance().getContext();
            SharedPreferences preferences = context.getSharedPreferences(USER_PROFILE_PREFERENCES, Context.MODE_PRIVATE);
            String profileId = preferences.getString(PROFILE_ID, null);
            if (profileId != null) {
                singleInstance = new UserProfile(context, profileId);
            }
        }
        return singleInstance;
    }

    public synchronized static UserProfile createNewProfile(@NonNull String identifyId) {
        return singleInstance = new UserProfile(identifyId);
    }

    public synchronized static UserProfile createNewProfile() {
        return singleInstance = new UserProfile(null);
    }


    private UserProfile(@Nullable String identifyId) {
        this.context = UserKit.getInstance().getContext();
        if (identifyId != null)
            this.profileId = identifyId;
        else
            initDefaultValue();
        saveUserProfile();
    }

    private UserProfile(Context context, String profileId) {
        this.context = context;
        this.profileId = profileId;
    }

    private void initDefaultValue() {
        String android_id = Settings.Secure.getString(context.getContentResolver(),
                Settings.Secure.ANDROID_ID);
        this.profileId = Utils.md5("android_" + android_id + "_" + new Date().getTime());
    }

    protected synchronized void saveUserProfile() {
        SharedPreferences preferences = context.getSharedPreferences(USER_PROFILE_PREFERENCES, Context.MODE_PRIVATE);
        preferences.edit()
                .putString(PROFILE_ID, profileId)
                .apply();
    }

    public void identify(@NonNull String id) {
        this.profileId = id;
        saveUserProfile();
    }

    public String getIdentifiedId() {
        return profileId;
    }

    public void mapProfileId(String profileId) {
        Intent intent = new Intent(context, SenderService.class);
        intent.putExtra(SenderService.ALIAS_ID, profileId);
        context.startService(intent);
    }

    /**
     * Key set.
     *
     * @param keys key set, separate by comma
     */
    public void getProperties(String keys, final GetPropertiesCallback callback) {
        UserKitService userKitService = ServiceGenerator.retrofit(Constants.USER_KIT_CLIENT)
                .create(UserKitService.class);
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        userKitService.getProperties(profileId, keys).enqueue(new Callback<Properties>() {
            @Override
            public void onResponse(Call<Properties> call, Response<Properties> response) {
                if (response.isSuccessful()) {
                    callback.onResponse(response.body());
                } else {
                    ErrorResponse errorResponse = ErrorHandler.parseError(response, Constants.USER_KIT_CLIENT);
                    callback.onError(new ServerResponseCodeException(response.code(), errorResponse));
                }
            }

            @Override
            public void onFailure(Call<Properties> call, Throwable t) {
                callback.onError(t);
            }
        });
    }

    public void getProperties(final GetPropertiesCallback callback, String... keyArray) {
        if (keyArray == null || keyArray.length == 0)
            throw new IllegalArgumentException("Key array must not be empty or null.");
        String keys = "";
        for (int i = 0; i < keyArray.length - 1; i++) {
            keys += keyArray[i] + ",";
        }
        keys += keyArray[keyArray.length - 1];
        getProperties(keys, callback);
    }

    public void getProperties(AbstractList<String> keyArray, final GetPropertiesCallback callback) {
        if (keyArray == null || keyArray.size() == 0)
            throw new IllegalArgumentException("Key array must not be empty or null.");
        String keys = "";
        for (int i = 0; i < keyArray.size() - 1; i++) {
            keys += keyArray.get(i) + ",";
        }
        keys += keyArray.get(keyArray.size() - 1);
        getProperties(keys, callback);
    }

    public ProfileEditor edit() {
        return new ProfileEditor();
    }


}
