package userkit.sdk.job;

import android.os.Parcel;
import android.util.Log;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Properties;

import retrofit2.Response;
import userkit.sdk.Constants;
import userkit.sdk.api.ErrorHandler;
import userkit.sdk.api.ErrorResponse;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.data.ProfileUpdatePack;
import userkit.sdk.exception.ServerResponseCodeException;
import userkit.sdk.exception.UserProfileNullException;
import userkit.sdk.profile.ProfilePropertyEditorBase;
import userkit.sdk.profile.UserProfile;
import userkit.sdk.profile.annotations.ProfileEditType;

/**
 * Created by khangnt on 8/22/16.
 * Email: khang.neon.1997@gmail.com
 */
public class PostProfileJob extends JobBase {
    private static final String TAG = "PostProfileJob";
    public static final double MAX_RETRY_TIMES = 2;

    @ProfileEditType
    String profileEditType;
    Properties properties;

    public PostProfileJob(ProfilePropertyEditorBase profileUpdateInfo) {
        super(Priority.MEDIUM, JobGroup.PROFILE_GROUP);
        profileEditType = profileUpdateInfo.getEditType();
        properties = profileUpdateInfo.getProperties();
    }

    public PostProfileJob(Parcel in) {
        super(in);
        //noinspection WrongConstant
        profileEditType = in.readString();
        String propertiesStr = in.readString();
        StringReader stringReader = new StringReader(propertiesStr);
        properties = new Properties();
        try {
            properties.load(stringReader);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void onRun() throws Throwable {
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        ProfileUpdatePack profileUpdatePack = new ProfileUpdatePack(profileId, properties);
        UserKitService userKitService = ServiceGenerator.retrofit(Constants.USER_KIT_CLIENT)
                .create(UserKitService.class);
        Response<Void> response = userKitService.profileUpdate(profileEditType, profileUpdatePack).execute();
        if (response.isSuccessful())
            Log.i(TAG, "Update profile success");
        else {
            ErrorResponse errorResponse = ErrorHandler.parseError(response, Constants.USER_KIT_CLIENT);
            // Throw server response exception
            throw new ServerResponseCodeException(response.code(), errorResponse);
        }
    }


    @Override
    protected int shouldRetryOnThrowException(Throwable exception, int runCount) {
        exception.printStackTrace();
        if (exception instanceof ServerResponseCodeException        //  Server response error, do not try again.
                || exception instanceof UserProfileNullException)   //  User profile = null
            return -1;
        return (runCount - 1 < MAX_RETRY_TIMES) ? 3000 : -1;
    }

    @Override
    public boolean canMergeJob(JobBase job) {
        return false;                           // Can not merge two profile update task.
    }

    @Override
    public void writeToParcel(Parcel out) {
        out.writeString(profileEditType);

        StringWriter propertiesWriter = new StringWriter();
        try {
            properties.store(propertiesWriter, null);
        } catch (IOException e) {
            e.printStackTrace();
        }
        String propertiesStr = propertiesWriter.toString();
        out.writeString(propertiesStr);
    }
}
