package userkit.sdk.job;

import android.os.Parcel;
import android.util.Log;

import java.util.ArrayList;
import java.util.List;

import retrofit2.Response;
import userkit.sdk.Constants;
import userkit.sdk.api.ErrorHandler;
import userkit.sdk.api.ErrorResponse;
import userkit.sdk.api.ServiceGenerator;
import userkit.sdk.api.UserKitService;
import userkit.sdk.data.Event;
import userkit.sdk.data.EventsPack;
import userkit.sdk.exception.ServerResponseCodeException;
import userkit.sdk.exception.UserProfileNullException;
import userkit.sdk.profile.UserProfile;

/**
 * Created by khangnt on 8/22/16.
 * Email: khang.neon.1997@gmail.com
 */
public class PostEventJob extends JobBase {
    public static final int MAX_EVENTS_PER_TIME = 20;
    public static final int MAX_RETRY_TIMES = 1;
    private static final String TAG = "PostEventJob";

    private List<Event> eventList;

    public PostEventJob(Event event) {
        super(Priority.LOW, JobGroup.EVENT_GROUP);
        eventList = new ArrayList<>();
        eventList.add(event);
    }

    protected PostEventJob(Parcel in) {
        super(in);
        eventList = new ArrayList<>();
        in.readTypedList(eventList, Event.CREATOR);
    }

    @Override
    protected void onRun() throws Throwable {
        UserProfile userProfile = UserProfile.getCurrentProfile();
        if (userProfile == null)
            throw new UserProfileNullException();
        String profileId = userProfile.getIdentifiedId();
        EventsPack eventsPack = new EventsPack(profileId, eventList);
        UserKitService userKitService = ServiceGenerator.retrofit(Constants.USER_KIT_CLIENT)
                .create(UserKitService.class);
        Response<Void> response = userKitService.postEvents(eventsPack).execute();
        if (response.isSuccessful())
            Log.i(TAG, "Post event success. Count: " + eventList.size());
        else {
            ErrorResponse errorResponse = ErrorHandler.parseError(response, Constants.USER_KIT_CLIENT);
            // Throw server response exception
            throw new ServerResponseCodeException(response.code(), errorResponse);
        }
    }

    @Override
    protected int shouldRetryOnThrowException(Throwable exception, int runCount) {
        exception.printStackTrace();
        if (exception instanceof ServerResponseCodeException        //  Server response error, do not try again.
                || exception instanceof UserProfileNullException)   //  User profile = null
            return -1;
        return (runCount - 1 < MAX_RETRY_TIMES) ? 3000 : -1;
    }

    @Override
    public boolean canMergeJob(JobBase job) {
        if (job instanceof PostEventJob) {
            PostEventJob secondJob = ((PostEventJob) job);
            if (eventList.size() + secondJob.getEventList().size() > MAX_EVENTS_PER_TIME)
                return false;
            eventList.addAll(new ArrayList<>(secondJob.getEventList()));
            return true;
        }
        return false;
    }

    @Override
    public void writeToParcel(Parcel out) {
        out.writeTypedList(eventList);
    }

    protected List<Event> getEventList() {
        return eventList;
    }
}
