package userkit.sdk.identity;


import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.support.annotation.Nullable;
import android.util.Base64;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;

import okhttp3.ResponseBody;
import retrofit2.Call;
import userkit.sdk.identity.api.HandleResponse;
import userkit.sdk.identity.api.ServiceGenerator;
import userkit.sdk.identity.api.Transformer;
import userkit.sdk.identity.callback.ICallback;
import userkit.sdk.identity.data.LoginResponse;
import userkit.sdk.identity.data.SignInEmailPasswordData;
import userkit.sdk.identity.data.SignInSocialData;
import userkit.sdk.identity.data.SignUpData;
import userkit.sdk.identity.data.TokenResponse;

/**
 * Created by khangnt on 8/24/16.
 * Email: khang.neon.1997@gmail.com
 */
public class UserKitIdentity {
    private static final String TAG = "UserKitIdentity";
    public static final String USER_KIT_API_TOKEN = "USER_KIT_API_TOKEN";
    public static final String USERKIT_IDENTITY_PREFERENCE = "userkit_preference";
    public static final String AUTH_TOKEN = "auth_token";

    private static Context context;
    private static String apiToken;

    public synchronized static void init(Context context) {
        init(context, null);
    }

    public synchronized static void init(Context context, String apiToken) {
        UserKitIdentity.context = context.getApplicationContext();
        if (apiToken != null)
            UserKitIdentity.setApiToken(apiToken);
        else {                  // Find api token from manifest
            try {
                ApplicationInfo ai = context.getPackageManager().getApplicationInfo(context.getPackageName(), PackageManager.GET_META_DATA);
                Bundle bundle = ai.metaData;
                apiToken = bundle.getString(USER_KIT_API_TOKEN);
                setApiToken(apiToken);
            } catch (PackageManager.NameNotFoundException ignore) {
            }
        }
    }

    public static void setApiToken(String apiToken) {
        UserKitIdentity.apiToken = apiToken;
    }

    public static String getApiToken() {
        if (apiToken == null)
            Log.w(TAG, "Warning: You haven't set API Token");
        return apiToken;
    }

    public Context getContext() {
        return context;
    }

    public static void setAuthToken(String authToken) {
        saveAuthToken(authToken);
    }

    public static String getAuthToken() {
        return loadAuthToken();
    }

    private static void saveAuthToken(String token) {
        SharedPreferences sharedPreferences = context.getSharedPreferences(USERKIT_IDENTITY_PREFERENCE, Context.MODE_PRIVATE);
        String encodedKey = null;
        if (token != null)
            encodedKey = Base64.encodeToString(token.getBytes(), Base64.DEFAULT);
        sharedPreferences.edit()
                .putString(AUTH_TOKEN, encodedKey)
                .apply();
    }

    @Nullable
    private static String loadAuthToken() {
        SharedPreferences sharedPreferences = context.getSharedPreferences(USERKIT_IDENTITY_PREFERENCE, Context.MODE_PRIVATE);
        String encodedAuthToken = sharedPreferences.getString(AUTH_TOKEN, null);
        if (encodedAuthToken != null) {
            try {
                return new String(Base64.decode(encodedAuthToken, Base64.DEFAULT), "UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
        }
        return null;
    }

    public static void signUp(SignUpData data, final ICallback<LoginResponse> callback) {
        Call<ResponseBody> call = ServiceGenerator.buildIdentityService(Constants.USER_KIT_IDENTITY)
                .signUp(Utils.buildRequestBody(data));
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, LoginResponse>() {
            @Override
            public LoginResponse transform(ResponseBody from) throws Exception {
                return new LoginResponse(from.string());
            }
        });
    }

    public static void signInEmailPassword(SignInEmailPasswordData data, final ICallback<LoginResponse> callback) {
        Call<ResponseBody> call = ServiceGenerator.buildIdentityService(Constants.USER_KIT_IDENTITY)
                .signInEmailPassword(data);
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, LoginResponse>() {
            @Override
            public LoginResponse transform(ResponseBody from) throws Exception {
                return new LoginResponse(from.string());
            }
        });
    }

    public static void signInFacebook(String fbAuthToken, final ICallback<LoginResponse> callback) {
        Call<ResponseBody> call = ServiceGenerator.buildIdentityService(Constants.USER_KIT_IDENTITY)
                .signInFacebook(new SignInSocialData(fbAuthToken));
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, LoginResponse>() {
            @Override
            public LoginResponse transform(ResponseBody from) throws Exception {
                return new LoginResponse(from.string());
            }
        });
    }

    public static void signInGooglePlus(String ggAuthToken, final ICallback<LoginResponse> callback) {
        Call<ResponseBody> call = ServiceGenerator.buildIdentityService(Constants.USER_KIT_IDENTITY)
                .signInGoogle(new SignInSocialData(ggAuthToken));
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, LoginResponse>() {
            @Override
            public LoginResponse transform(ResponseBody from) throws Exception {
                return new LoginResponse(from.string());
            }
        });
    }

    public static void createProfiles(JSONArray profiles, final ICallback<JSONArray> callback) throws JSONException {
        JSONObject data = new JSONObject();
        data.put("profiles", profiles);
        Call<ResponseBody> call = ServiceGenerator.buildProfilesService(Constants.USER_KIT_IDENTITY)
                .createProfiles(Utils.buildRequestBody(data));
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, JSONArray>() {
            @Override
            public JSONArray transform(ResponseBody from) throws Exception {
                return new JSONArray(from.string());
            }
        });
    }

    public static void getProfiles(final ICallback<JSONArray> callback) {
        Call<ResponseBody> call = ServiceGenerator.buildProfilesService(Constants.USER_KIT_IDENTITY)
                .getProfiles();
        HandleResponse.enqueueCallback(call, callback, new Transformer<ResponseBody, JSONArray>() {
            @Override
            public JSONArray transform(ResponseBody from) throws Exception {
                return new JSONArray(from.string());
            }
        });
    }

    public static void switchProfile(String profileId, final ICallback<TokenResponse> callback) {
        Call<TokenResponse> call = ServiceGenerator.buildProfilesService(Constants.USER_KIT_IDENTITY)
                .switchProfile(profileId);
        HandleResponse.enqueueCallback(call, callback);
    }
}
