/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.widget.adapter;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;

/**
 * A simple registry that may be used by data set adapters in order to track registered
 * {@link OnDataSetListener OnDataSetListeners} along with {@link OnDataSetActionListener OnDataSetActionListeners}
 * and also to fire callbacks for those listeners.
 *
 * @author Martin Albedinsky
 */
public final class DataSetListeners {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "DataSetListeners";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Adapter to which is this listeners registry attached.
	 */
	private final DataSet mDataSet;

	/**
	 * List containing all {@link OnDataSetListener OnDataSetListeners} that has been registered via
	 * {@link #registerOnDataSetListener(OnDataSetListener)}.
	 */
	private List<OnDataSetListener> mListeners;

	/**
	 * List containing all {@link OnDataSetActionListener OnDataSetActionListeners} that has been
	 * registered via {@link #registerOnDataSetActionListener(OnDataSetActionListener)}.
	 */
	private List<OnDataSetActionListener> mActionListeners;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new DataSetListeners registry for the given <var>dataSet</var>.
	 *
	 * @param dataSet The data set for which to create the new listeners registry.
	 */
	public DataSetListeners(@NonNull final DataSet dataSet) {
		this.mDataSet = dataSet;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Registers a callback to be invoked whenever {@link #notifyDataSetChanged()} or {@link #notifyDataSetInvalidated()}
	 * is called upon this listeners registry.
	 *
	 * @param listener The desired listener to register.
	 * @see #unregisterOnDataSetListener(OnDataSetListener)
	 */
	public void registerOnDataSetListener(@NonNull final OnDataSetListener listener) {
		if (mListeners == null) this.mListeners = new ArrayList<>(1);
		if (!mListeners.contains(listener)) this.mListeners.add(listener);
	}

	/**
	 * Notifies all registered {@link OnDataSetListener OnDataSetListeners} that the associated data
	 * set has been changed.
	 *
	 * @see #notifyDataSetInvalidated()
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void notifyDataSetChanged() {
		if (mListeners != null && !mListeners.isEmpty()) {
			for (final OnDataSetListener listener : mListeners) {
				listener.onDataSetChanged();
			}
		}
	}

	/**
	 * Notifies all registered {@link OnDataSetListener OnDataSetListeners} that the associated data
	 * set has been invalidated.
	 *
	 * @see #notifyDataSetChanged()
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void notifyDataSetInvalidated() {
		if (mListeners != null && !mListeners.isEmpty()) {
			for (final OnDataSetListener listener : mListeners) {
				listener.onDataSetInvalidated();
			}
		}
	}

	/**
	 * Unregisters the given <var>listener</var> from this listeners registry, so it will not receive
	 * any callbacks further.
	 *
	 * @param listener The desired listener to unregister.
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void unregisterOnDataSetListener(@NonNull final OnDataSetListener listener) {
		if (mListeners != null) this.mListeners.remove(listener);
	}

	/**
	 * Registers a callback to be invoked whenever {@link #notifyDataSetActionSelected(int, int, Object)}
	 * is called upon this listeners registry.
	 *
	 * @param listener The desired listener to register.
	 * @see #notifyDataSetActionSelected(int, int, Object)
	 * @see #unregisterOnDataSetActionListener(OnDataSetActionListener)
	 */
	public void registerOnDataSetActionListener(@NonNull final OnDataSetActionListener listener) {
		if (mActionListeners == null) this.mActionListeners = new ArrayList<>(1);
		if (!mActionListeners.contains(listener)) this.mActionListeners.add(listener);
	}

	/**
	 * Notifies all registered {@link OnDataSetActionListener OnDataSetActionListeners} that the
	 * specified <var>action</var> has been selected in the associated data set.
	 *
	 * @param action   The action that was selected.
	 * @param position The position for which was the specified action selected.
	 * @param payload  Additional payload data for the selected action. May be {@code null} if no
	 *                 payload has been specified.
	 * @return {@code True} if the action has been handled by one of the registered listeners,
	 * {@code false} otherwise.
	 * @see #registerOnDataSetActionListener(OnDataSetActionListener)
	 */
	public boolean notifyDataSetActionSelected(final int action, final int position, final Object payload) {
		if (mActionListeners != null && !mActionListeners.isEmpty()) {
			final long itemId = mDataSet.getItemId(position);
			for (final OnDataSetActionListener listener : mActionListeners) {
				if (listener.onDataSetActionSelected(action, position, itemId, payload)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Unregister the given <var>callback</var> from this listeners registry, so it will not receive
	 * any callbacks further.
	 *
	 * @param listener The desired listener to unregister.
	 * @see #registerOnDataSetActionListener(OnDataSetActionListener)
	 */
	public void unregisterOnDataSetActionListener(@NonNull final OnDataSetActionListener listener) {
		if (mActionListeners != null) this.mActionListeners.remove(listener);
	}

	/**
	 * Checks whether this listeners registry is empty or not.
	 *
	 * @return {@code True} if there are not listeners registered, {@code false} if there is at least
	 * one listener registered.
	 */
	public boolean isEmpty() {
		return (mListeners == null || mListeners.isEmpty()) && (mActionListeners == null || mActionListeners.isEmpty());
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
