/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.widget.adapter;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.ArrayList;
import java.util.List;

/**
 * A simple registry that may be used by data set adapters of which data set may be swapped, for a
 * new one, in order to track registered {@link OnDataSetSwapListener OnDataSetSwapListenera} and
 * also to fire callbacks for those listeners.
 *
 * @param <A> Type of the adapter that will use this listeners registry.
 * @param <D> Type of data set that may be swapped for the adapter.
 * @author Martin Albedinsky
 */
public final class SwappableDataSetAdapterListeners<A, D> {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "SwappableDataSetAdapterListeners";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Adapter to which is this listeners registry attached.
	 */
	private final A mAdapter;

	/**
	 * List containing all {@link OnDataSetSwapListener} that has been registered via
	 * {@link #registerOnDataSetSwapListener(OnDataSetSwapListener)}.
	 */
	private List<OnDataSetSwapListener<A, D>> mDataSetSwapListeners;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new SwappableDataSetAdapterListeners registry for the given <var>adapter</var>.
	 *
	 * @param adapter The adapter for which to create the new listeners registry.
	 */
	public SwappableDataSetAdapterListeners(@NonNull final A adapter) {
		this.mAdapter = adapter;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Registers a callback to be invoked whenever {@link #notifyDataSetSwapStarted(Object)} or
	 * {@link #notifyDataSetSwapFinished(Object)} is called upon this listeners registry.
	 *
	 * @param listener The desired listener to register.
	 * @see #unregisterOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	public void registerOnDataSetSwapListener(@NonNull OnDataSetSwapListener<A, D> listener) {
		if (this.mDataSetSwapListeners == null) this.mDataSetSwapListeners = new ArrayList<>(1);
		if (!mDataSetSwapListeners.contains(listener)) this.mDataSetSwapListeners.add(listener);
	}

	/**
	 * Notifies all registered {@link OnDataSetSwapListener OnDataSetSwapListeners} that the specified
	 * <var>dataSet</var> is about to be swapped for the associated adapter.
	 *
	 * @param dataSet The data set that is about to be swapped for the adapter.
	 * @see #notifyDataSetSwapFinished(Object)
	 * @see #registerOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	protected void notifyDataSetSwapStarted(@Nullable D dataSet) {
		if (mDataSetSwapListeners != null && !mDataSetSwapListeners.isEmpty()) {
			for (final OnDataSetSwapListener<A, D> listener : mDataSetSwapListeners) {
				listener.onDataSetSwapStarted(mAdapter, dataSet);
			}
		}
	}

	/**
	 * Notifies all registered {@link OnDataSetSwapListener OnDataSetSwapListeners} that the specified
	 * <var>dataSet</var> has been swapped for the associated adapter.
	 *
	 * @param dataSet The data set that has been swapped for the adapter.
	 * @see #notifyDataSetSwapStarted(Object)
	 * @see #registerOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	protected void notifyDataSetSwapFinished(@Nullable D dataSet) {
		if (mDataSetSwapListeners != null && !mDataSetSwapListeners.isEmpty()) {
			for (final OnDataSetSwapListener<A, D> listener : mDataSetSwapListeners) {
				listener.onDataSetSwapFinished(mAdapter, dataSet);
			}
		}
	}

	/**
	 * Unregisters the given <var>listener</var> from this listeners registry, so it will not receive
	 * any callbacks further.
	 *
	 * @param listener The desired listener to unregister.
	 * @see #registerOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	public void unregisterOnDataSetSwapListener(@NonNull OnDataSetSwapListener<A, D> listener) {
		if (mDataSetSwapListeners != null) this.mDataSetSwapListeners.remove(listener);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
