/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.widget.adapter;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.Arrays;
import java.util.List;

import universum.studios.android.widget.adapter.holder.AdapterHolder;
import universum.studios.android.widget.adapter.holder.ViewHolder;
import universum.studios.android.widget.adapter.holder.ViewHolderBinders;
import universum.studios.android.widget.adapter.holder.ViewHolderFactories;

/**
 * A {@link BaseListAdapter} and {@link SimpleAdapter} implementation which provides API for simple
 * management of items data set.
 *
 * <h3>Implementation</h3>
 * Inheritance hierarchies of this adapter class are not required to implement any methods except
 * {@link #onCreateViewHolder(android.view.ViewGroup, int)} or {@link #onBindViewHolder(ViewHolder, int)}
 * if the creation and binding logic provided by default holder factory and default holder binder
 * (see section below) are not desired and no custom factory nor binder are specified.
 *
 * <h3>Default Holder Factory and Binder</h3>
 * This adapter uses implementation of {@link AdapterHolder.Factory} provided via
 * {@link ViewHolderFactories#simple()} as its default holder factory and implementation of
 * {@link AdapterHolder.Binder} provided via {@link ViewHolderBinders#simple()} as its default holder
 * binder.
 *
 * @author Martin Albedinsky
 */
public class SimpleListAdapter<A extends SimpleListAdapter<A, VH, I>, VH extends ViewHolder, I>
		extends BaseListAdapter<A, VH, I>
		implements
		SimpleAdapter<A, I> {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "SimpleListAdapter";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Data set containing current data of this adapter.
	 */
	private final SimpleAdapterDataSet<I> mDataSet;

	/**
	 * Registry with {@link OnDataSetSwapListener OnDataSetSwapListeners} that are attached to this
	 * adapter.
	 */
	private SwappableDataSetAdapterListeners<A, List<I>> mListeners;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Same as {@link #SimpleListAdapter(Context, List)} for array of initial <var>items</var> data set.
	 */
	public SimpleListAdapter(@NonNull final Context context, @NonNull final I[] items) {
		this(context, Arrays.asList(items));
	}

	/**
	 * Creates a new instance of SimpleListAdapter with the given initial <var>items</var> data set.
	 *
	 * @param context Context in which will be this adapter used.
	 * @param items   List of items to be used as initial data set for this adapter.
	 */
	public SimpleListAdapter(@NonNull final Context context, @NonNull final List<I> items) {
		this(context);
		mDataSet.swapItems(items);
	}

	/**
	 * Creates a new instance of SimpleListAdapter with empty data set.
	 *
	 * @param context Context in which will be this adapter used.
	 * @see #SimpleListAdapter(Context, Object[])
	 * @see #SimpleListAdapter(Context, List)
	 */
	@SuppressWarnings("unchecked")
	public SimpleListAdapter(@NonNull final Context context) {
		super(context);
		this.mDataSet = new SimpleAdapterDataSet<>();
		this.mDataSet.setItemsCallback(new SimpleAdapterDataSet.SimpleItemsCallback() {

			/**
			 */
			@Override
			void onItemsChanged() {
				notifyDataSetChanged();
			}
		});
		this.mDataSet.setItemPositionResolver(new DataSetItemPositionResolver(this));
		setHolderFactory((AdapterHolder.Factory<VH>) ViewHolderFactories.simple());
		setHolderBinder(ViewHolderBinders.<A, VH>simple());
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 */
	@Override
	@SuppressWarnings("unchecked")
	public void registerOnDataSetSwapListener(@NonNull OnDataSetSwapListener<A, List<I>> listener) {
		if (mListeners == null) {
			this.mListeners = new SwappableDataSetAdapterListeners<>((A) this);
		}
		this.mListeners.registerOnDataSetSwapListener(listener);
	}

	/**
	 */
	@Override
	public void unregisterOnDataSetSwapListener(@NonNull OnDataSetSwapListener<A, List<I>> listener) {
		if (mListeners != null) this.mListeners.unregisterOnDataSetSwapListener(listener);
	}

	/**
	 */
	@Override
	public void changeItems(@Nullable final List<I> items) {
		swapItems(items);
	}

	/**
	 */
	@Nullable
	@Override
	public List<I> swapItems(@Nullable final List<I> items) {
		notifyItemsSwapStarted(items);
		final List<I> oldItems = onSwapItems(items);
		notifyItemsSwapFinished(items);
		notifyDataSetChanged();
		return oldItems;
	}

	/**
	 * Notifies all registered {@link OnDataSetSwapListener OnDataSetSwapListeners} that swapping
	 * of items for this adapter has been started.
	 *
	 * @param items The items that are about to be swapped for this adapter.
	 * @see #notifyItemsSwapFinished(List)
	 * @see #registerOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	protected final void notifyItemsSwapStarted(@Nullable List<I> items) {
		if (mListeners != null) this.mListeners.notifyDataSetSwapStarted(items);
	}

	/**
	 * Called from {@link #swapItems(List)} in order to swap data set of items of this adapter to
	 * the given <var>items</var>.
	 *
	 * @param items The new items data set to be changed for this adapter.
	 * @return Old items data set that has been attached to this adapter before. May be {@code null}
	 * if there were no items attached.
	 */
	@Nullable
	protected List<I> onSwapItems(@Nullable final List<I> items) {
		return mDataSet.swapItems(items);
	}

	/**
	 * Notifies all registered {@link OnDataSetSwapListener OnDataSetSwapListeners} that swapping
	 * of items for this adapter has been finished.
	 *
	 * @param items The items that has been swapped for this adapter.
	 * @see #notifyItemsSwapStarted(List)
	 * @see #registerOnDataSetSwapListener(OnDataSetSwapListener)
	 */
	protected final void notifyItemsSwapFinished(@Nullable List<I> items) {
		if (mListeners != null) this.mListeners.notifyDataSetSwapFinished(items);
	}

	/**
	 */
	@Override
	public void insertItem(@NonNull final I item) {
		mDataSet.insertItem(item);
	}

	/**
	 */
	@Override
	public void insertItem(final int position, @NonNull final I item) {
		mDataSet.insertItem(position, item);
	}

	/**
	 */
	@Override
	public void insertItems(@NonNull final List<I> items) {
		mDataSet.insertItems(items);
	}

	/**
	 */
	@Override
	public void insertItems(final int positionStart, @NonNull final List<I> items) {
		mDataSet.insertItems(positionStart, items);
	}

	/**
	 */
	@NonNull
	@Override
	public I swapItem(final int position, @NonNull final I item) {
		return mDataSet.swapItem(position, item);
	}

	/**
	 */
	@NonNull
	@Override
	public I swapItemById(final long itemId, @NonNull final I item) {
		return mDataSet.swapItemById(itemId, item);
	}

	/**
	 */
	@Override
	public void moveItem(final int fromPosition, final int toPosition) {
		mDataSet.moveItem(fromPosition, toPosition);
	}

	/**
	 */
	@NonNull
	@Override
	public I removeItem(final int position) {
		return mDataSet.removeItem(position);
	}

	/**
	 */
	@Override
	public int removeItem(@NonNull final I item) {
		return mDataSet.removeItem(item);
	}

	/**
	 */
	@NonNull
	@Override
	public I removeItemById(final long itemId) {
		return mDataSet.removeItemById(itemId);
	}

	/**
	 */
	@NonNull
	@Override
	public List<I> removeItems(final int positionStart, final int itemCount) {
		return mDataSet.removeItems(positionStart, itemCount);
	}

	/**
	 */
	@Nullable
	@Override
	public List<I> getItems() {
		return mDataSet.getItems();
	}

	/**
	 */
	@Override
	public int getItemCount() {
		return mDataSet.getItemCount();
	}

	/**
	 */
	@Override
	public boolean hasItemAt(final int position) {
		return mDataSet.hasItemAt(position);
	}

	/**
	 */
	@NonNull
	@Override
	public I getItem(final int position) {
		return mDataSet.getItem(position);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
