/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.widget.adapter;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;

import static junit.framework.Assert.assertTrue;

/**
 * A {@link SimpleDataSet} implementation that may be used to wrap a set of items and use it as data
 * set in simple adapter implementations.
 *
 * @author Martin Albedinsky
 */
final class SimpleAdapterDataSet<I> implements SimpleDataSet<I> {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	private static final String TAG = "SimpleAdapterDataSet";

	/*
	 * Interface ===================================================================================
	 */

	/**
	 * Interface which may be used to receive callbacks for <b>changed</b>, <b>inserted</b>,
	 * <b>moved</b> or <b>removed</b> items within associated {@link SimpleAdapterDataSet}.
	 */
	interface ItemsCallback {

		/**
		 * Invoked whenever a group of items has been inserted to the associated {@link SimpleAdapterDataSet}.
		 *
		 * @param positionStart Position of the first item that has been inserted.
		 * @param itemCount     Number of items inserted from the starting position.
		 */
		void onItemRangeInserted(int positionStart, int itemCount);

		/**
		 * Invoked whenever a group of items has been changed within the associated {@link SimpleAdapterDataSet}.
		 *
		 * @param positionStart Position of the first item that has been changed.
		 * @param itemCount     Number of items changed from the starting position.
		 */
		void onItemRangeChanged(int positionStart, int itemCount);

		/**
		 * Invoked whenever a single item has been moved within the associated {@link SimpleAdapterDataSet}.
		 *
		 * @param fromPosition The position from which has been item moved.
		 * @param toPosition   The position to which has been item moved.
		 */
		void onItemMoved(int fromPosition, int toPosition);

		/**
		 * Invoked whenever a group of items has been removed from the associated {@link SimpleAdapterDataSet}.
		 *
		 * @param positionStart Old position of the first item that has been removed.
		 * @param itemCount     Number of items removed from the starting position.
		 */
		void onItemRangeRemoved(int positionStart, int itemCount);
	}

	/**
	 * Simple implementation of ItemsCallback which may be used to handle all callbacks via single
	 * {@link SimpleItemsCallback#onItemsChanged()} callback.
	 */
	static class SimpleItemsCallback implements ItemsCallback {

		/**
		 */
		@Override
		public void onItemRangeInserted(int positionStart, int itemCount) {
			onItemsChanged();
		}

		/**
		 */
		@Override
		public void onItemRangeChanged(int positionStart, int itemCount) {
			onItemsChanged();
		}

		/**
		 */
		@Override
		public void onItemMoved(int fromPosition, int toPosition) {
			onItemsChanged();
		}

		/**
		 */
		@Override
		public void onItemRangeRemoved(int positionStart, int itemCount) {
			onItemsChanged();
		}

		/**
		 * Invoked whenever {@link #onItemMoved(int, int)} or one of {@code onItemRange...(...)}
		 * callbacks is received.
		 */
		void onItemsChanged() {
			// May be implemented by the inheritance hierarchies to handle change in data set of
			// items in a simple way.
		}
	}

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * List of items managed by this data set.
	 */
	private List<I> mItems;

	/**
	 * Callback fired whenever a change occurs in the items data set.
	 */
	private ItemsCallback mItemsCallback;

	/**
	 * Resolver instance which is used by this data set to resolve position of a specific item by
	 * its corresponding id.
	 */
	private ItemPositionResolver mItemPositionResolver;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new instance of SimpleAdapterDataSet without initial items.
	 */
	SimpleAdapterDataSet() {
		this(null);
	}

	/**
	 * Creates a new instance of SimpleAdapterDataSet with the given initial list of <var>items</var>.
	 *
	 * @param items The initial items for the new data set.
	 */
	SimpleAdapterDataSet(List<I> items) {
		this.mItems = items;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Registers a callback to be invoked whenever items managed by this data set change.
	 *
	 * @param callback The desired callback to register. May be {@code null} to clear the current one.
	 */
	void setItemsCallback(ItemsCallback callback) {
		this.mItemsCallback = callback;
	}

	/**
	 * Notifies the registered {@link ItemsCallback} that some of items of this data set has been
	 * inserted.
	 *
	 * @param positionStart Position of the first item that has been inserted.
	 * @param itemCount     Number of items that has been inserted from the starting position.
	 */
	private void notifyItemRangeInserted(int positionStart, int itemCount) {
		if (mItemsCallback != null) mItemsCallback.onItemRangeInserted(positionStart, itemCount);
	}

	/**
	 * Notifies the registered {@link ItemsCallback} that some of items of this data set has been
	 * changed.
	 *
	 * @param positionStart Position of the first item that has been changed.
	 * @param itemCount     Number of items that has been changed from the starting position.
	 */
	private void notifyItemRangeChanged(int positionStart, int itemCount) {
		if (mItemsCallback != null) mItemsCallback.onItemRangeChanged(positionStart, itemCount);
	}

	/**
	 * Notifies the registered {@link ItemsCallback} that a single item of this data set has been
	 * moved.
	 *
	 * @param fromPosition Position from which has been item moved.
	 * @param toPosition   Position to which has been item moved.
	 */
	private void notifyItemMoved(int fromPosition, int toPosition) {
		if (mItemsCallback != null) mItemsCallback.onItemMoved(fromPosition, toPosition);
	}

	/**
	 * Notifies the registered {@link ItemsCallback} that some of items of this data set has been
	 * removed.
	 *
	 * @param positionStart Old position of the first item that has been removed.
	 * @param itemCount     Number of items that has been removed from the starting position.
	 */
	private void notifyItemRangeRemoved(int positionStart, int itemCount) {
		if (mItemsCallback != null) mItemsCallback.onItemRangeRemoved(positionStart, itemCount);
	}

	/**
	 * Sets a resolver that will be used by this data set to resolve position of a specific item
	 * by its associated id.
	 * <p>
	 * <b>Note that this resolver is mainly used when there are called methods upon this data set
	 * taking as argument an item id like {@link #removeItemById(long)}</b>.
	 *
	 * @param resolver The desired resolver. May be {@code null} if no resolving should be done.
	 */
	void setItemPositionResolver(ItemPositionResolver resolver) {
		this.mItemPositionResolver = resolver;
	}

	/**
	 * Resolves position of an item associated with the specified <var>itemId</var>.
	 *
	 * @param itemId Id of the item of which position to resolve.
	 * @return Resolved position or {@link ItemPositionResolver#NO_POSITION} if there is no item
	 * position resolver attached or the attached resolver could not properly resolve the position.
	 * @see #setItemPositionResolver(ItemPositionResolver)
	 */
	private int resolvePositionForItemId(long itemId) {
		if (mItemPositionResolver == null) {
			throw new IllegalStateException("No position resolver for item id specified!");
		}
		return  mItemPositionResolver.resolveItemPosition(itemId);
	}

	/**
	 * Sets a new list of <var>items</var> to be managed by this data set and returns the old one.
	 *
	 * @param items The desired items to attach to this data set. May be {@code null}.
	 * @return Old list of items that has been attached to this data set before or {@code null} if
	 * there were no items attached.
	 * @see #getItems()
	 */
	List<I> swapItems(List<I> items) {
		final List<I> oldItems = mItems;
		this.mItems = items;
		return oldItems;
	}

	/**
	 * Returns the current list of items presented within this data set.
	 *
	 * @return Current items or {@code null} if there were no items attached nor added yet.
	 */
	List<I> getItems() {
		return mItems;
	}

	/**
	 */
	@Override
	public boolean isEmpty() {
		return getItemCount() == 0;
	}

	/**
	 */
	@Override
	public int getItemCount() {
		return mItems == null ? 0 : mItems.size();
	}

	/**
	 */
	@Override
	public void insertItem(@NonNull I item) {
		insertItem(getItemCount(), item);
	}

	/**
	 */
	@Override
	public void insertItem(int position, @NonNull I item) {
		if (mItems == null) {
			this.mItems = new ArrayList<>();
			this.mItems.add(item);
			AdaptersLogging.d(TAG, "Inserted item at position(0).");
			this.notifyItemRangeInserted(0, 1);
		} else {
			this.mItems.add(position, item);
			AdaptersLogging.d(TAG, "Inserted item at position(" + position + ").");
			this.notifyItemRangeInserted(position, 1);
		}
	}

	/**
	 */
	@Override
	public void insertItems(@NonNull List<I> items) {
		insertItems(getItemCount(), items);
	}

	/**
	 */
	@Override
	public void insertItems(int positionStart, @NonNull List<I> items) {
		final int itemsSize = items.size();
		if (mItems == null) {
			this.mItems = new ArrayList<>(items);
			AdaptersLogging.d(TAG, "Inserted items from start position(0) in count(" + itemsSize + ").");
			this.notifyItemRangeInserted(0, itemsSize);
		} else {
			this.mItems.addAll(positionStart, items);
			AdaptersLogging.d(TAG, "Inserted items from start position(" + positionStart + ") in count(" + itemsSize + ").");
			this.notifyItemRangeInserted(positionStart, itemsSize);
		}
	}

	/**
	 */
	@NonNull
	@Override
	public I swapItemById(long itemId, @NonNull I item) {
		final int position = resolvePositionForItemId(itemId);
		if (position == NO_POSITION) {
			throw new AssertionError("No item to be swapped with id(" + itemId + ") found!");
		}
		return swapItem(position, item);
	}

	/**
	 */
	@NonNull
	@Override
	public I swapItem(int position, @NonNull I item) {
		assertTrue("No item to be swapped at position(" + position + ") found!", hasItemAt(position));
		final I oldItem = mItems.set(position, item);
		AdaptersLogging.d(TAG, "Swapped item at position(" + position + ").");
		this.notifyItemRangeChanged(position, 1);
		return oldItem;
	}

	/**
	 */
	@Override
	public void moveItem(int fromPosition, int toPosition) {
		assertTrue("No item to be moved from position(" + fromPosition + ") to position(" + toPosition + ") found!", hasItemAt(fromPosition));
		assertTrue("No item to be moved from position(" + toPosition + ") to position(" + fromPosition + ") found!", hasItemAt(toPosition));
		mItems.set(fromPosition, mItems.set(toPosition, mItems.get(fromPosition)));
		AdaptersLogging.d(TAG, "Moved item from position(" + fromPosition + ") to position(" + toPosition + ").");
		this.notifyItemMoved(fromPosition, toPosition);
	}

	/**
	 */
	@Override
	public int removeItem(@NonNull I item) {
		final int position = mItems == null ? NO_POSITION : mItems.indexOf(item);
		if (position == NO_POSITION) {
			throw new AssertionError("No item(" + item + ") to be removed found!");
		}
		removeItem(position);
		return position;
	}

	/**
	 */
	@NonNull
	@Override
	public I removeItemById(long itemId) {
		final int position = resolvePositionForItemId(itemId);
		if (position == NO_POSITION) {
			throw new AssertionError("No item to be removed with id(" + itemId + ") found!");
		}
		return removeItem(position);
	}

	/**
	 */
	@NonNull
	@Override
	public I removeItem(int position) {
		if (hasItemAt(position)) {
			final I item = mItems.remove(position);
			AdaptersLogging.d(TAG, "Removed item at position(" + position + ").");
			this.notifyItemRangeRemoved(position, 1);
			return item;
		}
		throw new AssertionError("No item to be removed at position(" + position + ") found!");
	}

	/**
	 */
	@NonNull
	@Override
	public List<I> removeItems(int positionStart, int itemCount) {
		final int itemsSize = mItems.size();
		if (positionStart < 0 || positionStart + itemCount > itemsSize) {
			throw new IndexOutOfBoundsException(
					"Starting position(" + positionStart + ") and/or item count(" + itemCount + ") are out of range (positionStart < 0 || positionStart + itemCount > " + itemsSize + ")!"
			);
		}
		final List<I> items = new ArrayList<>(itemCount);
		for (int i = positionStart + itemCount - 1; i >= positionStart; i--) {
			items.add(0, mItems.remove(i));
		}
		AdaptersLogging.d(TAG, "Removed items from start position(" + positionStart + ") in count(" + itemCount + ").");
		this.notifyItemRangeRemoved(positionStart, itemCount);
		return items;
	}

	/**
	 */
	@Override
	public boolean hasItemAt(int position) {
		return position >= 0 && position < getItemCount();
	}

	/**
	 */
	@Override
	public long getItemId(int position) {
		return hasItemAt(position) ? position : NO_POSITION;
	}

	/**
	 */
	@NonNull
	@Override
	@SuppressWarnings({"unchecked", "ConstantConditions"})
	public I getItem(int position) {
		if (!hasItemAt(position)) {
			throw new IndexOutOfBoundsException(
					"Requested item at invalid position(" + position + "). " +
							"Data set has items in count of(" + getItemCount() + ")."
			);
		}
		return mItems.get(position);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
