/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.widget.adapter;

import android.support.annotation.NonNull;

import java.util.ArrayList;
import java.util.List;

/**
 * A simple registry that may be used by data set adapters in order to track registered
 * {@link OnDataSetListener OnDataSetListeners} along with {@link OnDataSetActionListener OnDataSetActionListeners}
 * and also to fire callbacks for those listeners.
 *
 * @param <A> Type of the adapter that will use this listeners registry.
 * @author Martin Albedinsky
 */
public final class DataSetAdapterListeners<A extends DataSetAdapter> {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "DataSetAdapterListeners";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Adapter to which is this listeners registry attached.
	 */
	private final A mAdapter;

	/**
	 * List containing all {@link OnDataSetListener OnDataSetListeners} that has been registered via
	 * {@link #registerOnDataSetListener(OnDataSetListener)}.
	 */
	private List<OnDataSetListener<A>> mDataSetListeners;

	/**
	 * List containing all {@link OnDataSetActionListener OnDataSetActionListeners} that has been
	 * registered via {@link #registerOnDataSetActionListener(OnDataSetActionListener)}.
	 */
	private List<OnDataSetActionListener<A>> mDataSetActionListeners;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new AdapterDataSetListeners registry for the given <var>adapter</var>.
	 *
	 * @param adapter The adapter for which to create the new listeners registry.
	 */
	public DataSetAdapterListeners(@NonNull final A adapter) {
		this.mAdapter = adapter;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Registers a callback to be invoked whenever {@link #notifyDataSetChanged()} or {@link #notifyDataSetInvalidated()}
	 * is called upon this listeners registry.
	 *
	 * @param listener The desired listener to register.
	 * @see #unregisterOnDataSetListener(OnDataSetListener)
	 */
	public void registerOnDataSetListener(@NonNull final OnDataSetListener<A> listener) {
		if (mDataSetListeners == null) this.mDataSetListeners = new ArrayList<>(1);
		if (!mDataSetListeners.contains(listener)) this.mDataSetListeners.add(listener);
	}

	/**
	 * Notifies all registered {@link OnDataSetListener OnDataSetListeners} that data set of the
	 * associated adapter has been changed.
	 *
	 * @see #notifyDataSetInvalidated()
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void notifyDataSetChanged() {
		if (mDataSetListeners != null && !mDataSetListeners.isEmpty()) {
			for (final OnDataSetListener<A> listener : mDataSetListeners) {
				listener.onDataSetChanged(mAdapter);
			}
		}
	}

	/**
	 * Notifies all registered {@link OnDataSetListener OnDataSetListeners} that data set of the
	 * associated adapter has been invalidated.
	 *
	 * @see #notifyDataSetChanged()
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void notifyDataSetInvalidated() {
		if (mDataSetListeners != null && !mDataSetListeners.isEmpty()) {
			for (final OnDataSetListener<A> listener : mDataSetListeners) {
				listener.onDataSetInvalidated(mAdapter);
			}
		}
	}

	/**
	 * Unregisters the given <var>listener</var> from this listeners registry, so it will not receive
	 * any callbacks further.
	 *
	 * @param listener The desired listener to unregister.
	 * @see #registerOnDataSetListener(OnDataSetListener)
	 */
	public void unregisterOnDataSetListener(@NonNull final OnDataSetListener<A> listener) {
		if (mDataSetListeners != null) this.mDataSetListeners.remove(listener);
	}

	/**
	 * Registers a callback to be invoked whenever {@link #notifyDataSetActionSelected(int, int, Object)}
	 * is called upon this listeners registry.
	 *
	 * @param listener The desired listener to register.
	 * @see #notifyDataSetActionSelected(int, int, Object)
	 * @see #unregisterOnDataSetActionListener(OnDataSetActionListener)
	 */
	public void registerOnDataSetActionListener(@NonNull final OnDataSetActionListener<A> listener) {
		if (mDataSetActionListeners == null) this.mDataSetActionListeners = new ArrayList<>(1);
		if (!mDataSetActionListeners.contains(listener)) this.mDataSetActionListeners.add(listener);
	}

	/**
	 * Notifies all registered {@link OnDataSetActionListener OnDataSetActionListeners} that the
	 * specified <var>action</var> has been selected in the data set of the associated adapter.
	 *
	 * @param action   The action that was selected.
	 * @param position The position for which was the specified action selected.
	 * @param payload  Additional payload data for the selected action. May be {@code null} if no
	 *                 payload has been specified.
	 * @return {@code True} if the action has been handled by one of the registered listeners,
	 * {@code false} otherwise.
	 * @see #registerOnDataSetActionListener(OnDataSetActionListener)
	 */
	public boolean notifyDataSetActionSelected(final int action, final int position, final Object payload) {
		if (mDataSetActionListeners != null && !mDataSetActionListeners.isEmpty()) {
			final long itemId = mAdapter.getItemId(position);
			for (final OnDataSetActionListener<A> listener : mDataSetActionListeners) {
				if (listener.onDataSetActionSelected(mAdapter, action, position, itemId, payload)) {
					return true;
				}
			}
		}
		return false;
	}

	/**
	 * Unregister the given <var>callback</var> from this listeners registry, so it will not receive
	 * any callbacks further.
	 *
	 * @param listener The desired listener to unregister.
	 * @see #registerOnDataSetActionListener(OnDataSetActionListener)
	 */
	public void unregisterOnDataSetActionListener(@NonNull final OnDataSetActionListener<A> listener) {
		if (mDataSetActionListeners != null) this.mDataSetActionListeners.remove(listener);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
