/*
 * *************************************************************************************************
 *                                 Copyright 2017 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.setting;

import android.content.res.Configuration;
import android.os.Bundle;
import android.preference.PreferenceActivity;
import android.support.annotation.CallSuper;
import android.support.annotation.ColorInt;
import android.support.annotation.IdRes;
import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.XmlRes;
import android.support.v7.app.ActionBar;
import android.support.v7.app.AppCompatDelegate;
import android.support.v7.widget.Toolbar;
import android.util.AndroidRuntimeException;
import android.view.LayoutInflater;
import android.view.MenuInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ListAdapter;
import android.widget.ListView;

import java.util.List;

/**
 * A {@link PreferenceActivity} implementation that uses {@link AppCompatDelegate} to provide
 * compatibility support features like attaching of custom action bar via {@link #setSupportActionBar(Toolbar)}.
 * <p>
 * This activity also intercepts default setting of list adapter with headers and uses {@link SettingHeadersAdapter}
 * to provide item views for the loaded preference headers via {@link #loadHeadersFromResource(int, List)}.
 * The headers adapter is specified via {@link #setHeadersAdapter(ListAdapter)} and the current one
 * may be obtained via {@link #setHeadersAdapter(ListAdapter)}. The headers that has been loaded and
 * are used by the headers adapter may be obtained via {@link #getHeaders()}.
 * <p>
 * If a {@link Toolbar} should be presented in the activity's view hierarchy, it may be added via
 * {@link #addToolbar()} which also attaches the added toolbar as support action bar via {@link #setSupportActionBar(Toolbar)}.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
public abstract class SettingsBaseActivity extends PreferenceActivity {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "SettingsBaseActivity";

	/**
	 * Bundle key used by parent {@link PreferenceActivity} to save and restore list of headers.
	 */
	private static final String BUNDLE_HEADERS = ":android:headers";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/**
	 * Corrector used by instances of SettingsBaseActivity to correct/modify theirs root layout to
	 * match actual Material design guidelines.
	 */
	private static final LayoutCorrector LAYOUT_CORRECTOR = new BasicLayoutCorrector();

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Application compatibility delegate used by this activity to provide compatibility features.
	 */
	private AppCompatDelegate compatDelegate;

	/**
	 * Current content view of this activity. Should be always valid whenever {@link #onContentChanged()}
	 * is received.
	 */
	private View contentView;

	/**
	 * Toolbar added into view hierarchy of this activity via {@link #addToolbar()}.
	 */
	private Toolbar toolbar;

	/**
	 * List containing preference headers inflated for this activity via {@link #loadHeadersFromResource(int, List)}.
	 */
	private List<Header> headers;

	/**
	 * Adapter providing item views for inflated preference headers.
	 *
	 * @see #setHeadersAdapter(ListAdapter)
	 */
	private ListAdapter headersAdapter;

	/*
	 * Constructors ================================================================================
	 */

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Returns the application compatibility delegate that is used by this activity to provide
	 * compatibility features like attaching of support action bar via {@link #setSupportActionBar(Toolbar)}.
	 *
	 * @return This activity's compatibility delegate.
	 */
	@NonNull protected final AppCompatDelegate delegate() {
		return compatDelegate == null ? (compatDelegate = AppCompatDelegate.create(this, null)) : compatDelegate;
	}

	/**
	 */
	@Override protected void onCreate(@Nullable final Bundle savedInstanceState) {
		delegate().installViewFactory();
		delegate().onCreate(savedInstanceState);
		super.onCreate(savedInstanceState);
		if (savedInstanceState != null) {
			this.headers = savedInstanceState.getParcelableArrayList(BUNDLE_HEADERS);
		}
		if (headers != null) {
			setHeadersAdapter(new SettingHeadersAdapter(this, headers));
		}
	}

	/**
	 */
	@Override public void loadHeadersFromResource(@XmlRes final int resid, @NonNull final List<Header> target) {
		super.loadHeadersFromResource(resid, this.headers = target);
	}

	/**
	 * Returns the list of headers that has been loaded via {@link #loadHeadersFromResource(int, List)}.
	 *
	 * @return The loaded list of headers or {@code null} if no headers has been loaded yet.
	 */
	@Nullable protected final List<Header> getHeaders() {
		return headers;
	}

	/**
	 * Sets an adapter that will provide item views for preference items to be presented in the
	 * view hierarchy of this activity.
	 *
	 * @param adapter The desired adapter providing headers for this activity's list view. May be
	 *                {@code null} to clear the current one.
	 */
	public void setHeadersAdapter(@Nullable final ListAdapter adapter) {
		super.setListAdapter(headersAdapter = adapter);
	}

	/**
	 * Returns the current headers adapter specified for this activity.
	 *
	 * @return This activity's headers adapter or {@code null} if there was no adapter specified yet.
	 * @see #setHeadersAdapter(ListAdapter)
	 */
	@Nullable public ListAdapter getHeadersAdapter() {
		return headersAdapter;
	}

	/**
	 * This method does nothing.
	 *
	 * @deprecated Use {@link #setHeadersAdapter(ListAdapter)} instead.
	 */
	@Deprecated @Override public final void setListAdapter(ListAdapter adapter) {
		// Ignored.
	}

	/**
	 * @return Always {@code null}.
	 * @deprecated Use {@link #getHeadersAdapter()} instead.
	 */
	@Deprecated @Override public final ListAdapter getListAdapter() {
		return null;
	}

	/**
	 */
	@Override protected void onPostCreate(@Nullable final Bundle savedInstanceState) {
		super.onPostCreate(savedInstanceState);
		delegate().onPostCreate(savedInstanceState);
	}

	/**
	 */
	@Override @NonNull public MenuInflater getMenuInflater() {
		return delegate().getMenuInflater();
	}

	/**
	 */
	@Override public void invalidateOptionsMenu() {
		delegate().invalidateOptionsMenu();
	}

	/**
	 */
	@Override public void setContentView(@LayoutRes final int layoutResID) {
		delegate().setContentView(layoutResID);
	}

	/**
	 */
	@Override public void setContentView(final View view) {
		delegate().setContentView(view);
	}

	/**
	 */
	@Override public void setContentView(final View view, final ViewGroup.LayoutParams params) {
		delegate().setContentView(view, params);
	}

	/**
	 */
	@Override public void addContentView(final View view, final ViewGroup.LayoutParams params) {
		delegate().addContentView(view, params);
	}

	/**
	 */
	@Override @CallSuper public void onContentChanged() {
		super.onContentChanged();
		this.contentView = ((ViewGroup) getWindow().getDecorView().findViewById(android.R.id.content)).getChildAt(0);
		LAYOUT_CORRECTOR.correctLayout(contentView);
	}

	/**
	 * Returns the content (root) view of this activity.
	 *
	 * @return This activity's content view.
	 * @throws IllegalStateException If content view of this activity has not been created yet.
	 * @see #onContentChanged()
	 */
	@NonNull protected final View getContentView() {
		if (contentView == null) {
			throw new IllegalStateException("Activity(" + getClass() + ") does not have content view created yet!");
		}
		return contentView;
	}

	/**
	 * Adds {@link Toolbar} into view hierarchy of this activity if it has not been added yet.
	 * <p>
	 * This implementation creates a new instance of the toolbar via {@link #onCreateToolbar(LayoutInflater, ViewGroup)}
	 * if it is not created and added yet and adds it into this activity's content view at the
	 * {@code 0} position, which is at the top. Also the added toolbar is attached to this activity
	 * as support action bar via {@link #setSupportActionBar(Toolbar)} and if this activity is not
	 * currently in state of hiding headers, the displaying of home as up for the attached action bar
	 * is enabled via {@link ActionBar#setDisplayHomeAsUpEnabled(boolean)} and {@link View.OnClickListener}
	 * is attached to the toolbar via {@link Toolbar#setNavigationOnClickListener(View.OnClickListener)}
	 * which when its {@link View.OnClickListener#onClick(View)} callback is fired calls
	 * {@link #onBackPressed()} of this activity.
	 *
	 * @return The added toolbar.
	 * @throws AndroidRuntimeException If the content view of this activity is not a ViewGroup, which
	 *                                 should not happen.
	 * @see #getToolbar()
	 * @see #getSupportActionBar()
	 */
	@SuppressWarnings("ConstantConditions")
	@NonNull protected Toolbar addToolbar() {
		if (!(contentView instanceof ViewGroup)) {
			throw new AndroidRuntimeException("Cannot add Toolbar. The content view of activity(" + getClass() + ") is not a ViewGroup!");
		}
		if (toolbar == null) {
			this.toolbar = onCreateToolbar(getLayoutInflater(), (ViewGroup) contentView);
			// Add Toolbar at the top of layout which is LinearLayout.
			((ViewGroup) contentView).addView(toolbar, 0);
			setSupportActionBar(toolbar);
			if (onIsHidingHeaders()) {
				final ActionBar actionBar = getSupportActionBar();
				actionBar.setDisplayHomeAsUpEnabled(true);
				this.toolbar.setNavigationOnClickListener(new View.OnClickListener() {

					/**
					 */
					@Override public void onClick(@NonNull final View view) {
						onBackPressed();
					}
				});
			}
		}
		return toolbar;
	}

	/**
	 * Invoked as result to call of {@link #addToolbar()} to create a new instance of {@link Toolbar}
	 * that will be added into view hierarchy of this activity.
	 *
	 * @param inflater Layout inflater that may be used to inflate the requested toolbar.
	 * @param root     This activity's root view.
	 * @return Instance of toolbar to be added into this activity's view hierarchy.
	 */
	@NonNull protected Toolbar onCreateToolbar(@NonNull final LayoutInflater inflater, @NonNull final ViewGroup root) {
		return new Toolbar(inflater.getContext());
	}

	/**
	 * Returns the toolbar added to the view hierarchy of this activity via {@link #addToolbar()}.
	 * <p>
	 * If added, this toolbar may be also obtained as support action bar via {@link #getSupportActionBar()}.
	 *
	 * @return This activity's toolbar if added, {@code null} otherwise.
	 */
	@Nullable protected Toolbar getToolbar() {
		return toolbar;
	}

	/**
	 */
	@Override public <V extends View> V findViewById(@IdRes final int id) {
		return delegate().findViewById(id);
	}

	/**
	 * Attaches the given <var>toolbar</var> to this activity as support action bar.
	 *
	 * @param toolbar The desired toolbar to attach as action bar. May be {@code null} to clear the
	 *                current one.
	 * @see #getSupportActionBar()
	 */
	public void setSupportActionBar(@Nullable final Toolbar toolbar) {
		delegate().setSupportActionBar(toolbar);
	}

	/**
	 * Returns the support action bar attached to this activity via {@link #setSupportActionBar(Toolbar)}.
	 *
	 * @return This activity's support action bar or {@code null} if there is no action bar attached.
	 */
	@Nullable public ActionBar getSupportActionBar() {
		return delegate().getSupportActionBar();
	}

	/**
	 */
	@Override protected void onPostResume() {
		super.onPostResume();
		delegate().onPostResume();
	}

	/**
	 */
	@Override protected void onListItemClick(@NonNull final ListView listView, @NonNull final View itemView, final int position, final long id) {
		if (!(headersAdapter instanceof SettingHeadersAdapter)) {
			super.onListItemClick(listView, itemView, position, id);
			return;
		}
		final SettingHeadersAdapter.Item item = (SettingHeadersAdapter.Item) headersAdapter.getItem(position);
		onHeaderClick(item.getHeader(), position);
	}

	/**
	 */
	@Override protected void onSaveInstanceState(@NonNull final Bundle state) {
		super.onSaveInstanceState(state);
		delegate().onSaveInstanceState(state);
	}

	/**
	 */
	@Override protected void onTitleChanged(final CharSequence title, @ColorInt final int color) {
		super.onTitleChanged(title, color);
		delegate().setTitle(title);
	}

	/**
	 */
	@Override public void onConfigurationChanged(@NonNull final Configuration newConfig) {
		super.onConfigurationChanged(newConfig);
		delegate().onConfigurationChanged(newConfig);
	}

	/**
	 */
	@Override protected void onStop() {
		super.onStop();
		delegate().onStop();
	}

	/**
	 */
	@Override protected void onDestroy() {
		super.onDestroy();
		delegate().onDestroy();
	}

	/*
	 * Inner classes ===============================================================================
	 */

	/**
	 * Interface for correctors used to correct/modify the view hierarchy of {@link SettingsBaseActivity}
	 * so it matches actual Material design guidelines.
	 */
	private interface LayoutCorrector {

		/**
		 * Performs correction/modification of the given <var>layout</var>.
		 *
		 * @param layout The root layout to be corrected/modified.
		 */
		void correctLayout(@NonNull View layout);
	}

	/**
	 * A {@link LayoutCorrector} basic implementation that performs layout corrections related to
	 * all Android platform versions.
	 */
	private static class BasicLayoutCorrector implements LayoutCorrector {

		/**
		 */
		@Override public void correctLayout(@NonNull final View layout) {
			if (layout instanceof ViewGroup) {
				final View firstChild = ((ViewGroup) layout).getChildAt(0);
				if (firstChild instanceof ViewGroup && ((ViewGroup) firstChild).getChildCount() > 1) {
					final ViewGroup panelsContainer = (ViewGroup) firstChild;
					correctHeadersPanel(panelsContainer.getChildAt(0));
					correctPreferencesPanel(panelsContainer.getChildAt(1));
				}
			}
		}

		/**
		 * Performs correction of the given <var>headersPanelView</var>.
		 * <p>
		 * This implementation clears padding of the given headers panel view and also of the headers
		 * list view presented in view hierarchy of the panel view.
		 *
		 * @param headersPanelView The headers panel view to be corrected.
		 */
		void correctHeadersPanel(final View headersPanelView) {
			headersPanelView.setPadding(0, 0, 0, 0);
			final ListView headersListView = (ListView) headersPanelView.findViewById(android.R.id.list);
			headersListView.setPadding(0, 0, 0, 0);
			headersListView.setDivider(null);
			headersListView.setDividerHeight(0);
		}

		/**
		 * Performs correction of the given <var>preferencesPanelView</var>.
		 * <p>
		 * This implementation clears padding of the given preferences panel view and also of the
		 * preferences frame view, if presented in the view hierarchy of the panel view.
		 *
		 * @param preferencesPanelView The preferences panel view to be corrected.
		 */
		void correctPreferencesPanel(final View preferencesPanelView) {
			preferencesPanelView.setPadding(0, 0, 0, 0);
			if (preferencesPanelView instanceof ViewGroup && ((ViewGroup) preferencesPanelView).getChildCount() > 0) {
				final ViewGroup preferencesPanelViewGroup = (ViewGroup) preferencesPanelView;
				final View preferencesFrameView = preferencesPanelViewGroup.getChildAt(preferencesPanelViewGroup.getChildCount() - 1);
				if (preferencesFrameView != null) {
					((ViewGroup) preferencesFrameView).setOnHierarchyChangeListener(new ViewGroup.OnHierarchyChangeListener() {

						/**
						 */
						@Override public void onChildViewAdded(@NonNull final View parent, @NonNull final View child) {
							preferencesFrameView.setPadding(0, 0, 0, 0);
						}

						/**
						 */
						@Override public void onChildViewRemoved(@NonNull final View parent, @NonNull final View child) {
							// Ignored.
						}
					});
				}
			}
		}
	}
}