/*
 * *************************************************************************************************
 *                                 Copyright 2017 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.preference;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

/**
 * Interface for simple provider which provides access to {@link SharedPreferences} for specified
 * {@link #setSharedPreferencesName(String) name} and {@link #setSharedPreferencesMode(int) mode}.
 * Instance of the corresponding shared preferences may be obtained via {@link #getSharedPreferences()}.
 *
 * @author Martin Albedinsky
 * @since 2.0
 */
public interface SharedPreferencesProvider {

	/**
	 * Sets a name for preferences file that is used by this provider to access instance of
	 * {@link SharedPreferences} via {@link android.content.Context#getSharedPreferences(String, int)}.
	 *
	 * @param name The desired preferences name. May be {@code null} if this host should use default
	 *             name provide by the framework.
	 *
	 * @see #getSharedPreferencesName()
	 * @see #setSharedPreferencesMode(int)
	 * @see #getSharedPreferences()
	 */
	void setSharedPreferencesName(@Nullable String name);

	/**
	 * Returns the name for preferences file specified for this provider.
	 * <p>
	 * Default value: {@link android.preference.PreferenceManager#getDefaultSharedPreferencesName(android.content.Context)
	 * PreferenceManager#getDefaultSharedPreferencesName(Context)}
	 *
	 * @return Preferences name. If no name has been specified a default one will be returned.
	 *
	 * @see #setSharedPreferencesName(String)
	 * @see #getSharedPreferencesMode()
	 * @see #getSharedPreferences()
	 */
	@NonNull String getSharedPreferencesName();

	/**
	 * Sets a mode for preferences file that is used by this provider to access instance of
	 * {@link SharedPreferences} via {@link android.content.Context#getSharedPreferences(String, int) Context.getSharedPreferences(String, int)}.
	 *
	 * @param mode The desired preferences mode.
	 *
	 * @see #getSharedPreferencesMode()
	 * @see #setSharedPreferencesName(String)
	 * @see #getSharedPreferences()
	 */
	void setSharedPreferencesMode(@SharedPreferencesPolicies.Mode int mode);

	/**
	 * Returns the mode for preferences file specified for this provider.
	 * <p>
	 * Default value: {@link SharedPreferencesPolicies#MODE_PRIVATE MODE_PRIVATE}
	 *
	 * @return Preferences mode. If no mode has been specified a default one will be returned.
	 *
	 * @see #setSharedPreferencesMode(int)
	 * @see #getSharedPreferencesName()
	 * @see #getSharedPreferences()
	 */
	@SharedPreferencesPolicies.Mode int getSharedPreferencesMode();

	/**
	 * Returns the instance of SharedPreferences provided by this provider.
	 *
	 * @return The associated shared preferences.
	 *
	 * @see #setSharedPreferencesName(String)
	 * @see #setSharedPreferencesMode(int)
	 */
	@NonNull SharedPreferences getSharedPreferences();
}