/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.preference;

import android.content.SharedPreferences;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import java.util.Map;
import java.util.Set;

/**
 * A {@link SharedPreferences} implementation which may be used to wrap an instance of {@link SharedPreferences}.
 * This wrapper class simply delegates all its calls to the wrapped preferences instance.
 *
 * @author Martin Albedinsky
 */
public class SharedPreferencesWrapper implements SharedPreferences {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "SharedPreferencesWrapper";

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Wrapped instance of SharedPreferences to which is this wrapper delegating all its calls.
	 */
	private final SharedPreferences mPreferences;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new instance of SharedPreferencesWrapper in order to wrap the given <var>preferences</var>
	 * instance.
	 *
	 * @param preferences The shared preferences to be wrapped.
	 */
	public SharedPreferencesWrapper(@NonNull final SharedPreferences preferences) {
		this.mPreferences = preferences;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Returns the wrapped SharedPreferences instance.
	 *
	 * @return Instance of SharedPreferences that are wrapped by this preferences wrapper.
	 */
	@NonNull
	public final SharedPreferences getWrappedPreferences() {
		return mPreferences;
	}

	/**
	 */
	@Override
	public void registerOnSharedPreferenceChangeListener(@NonNull final OnSharedPreferenceChangeListener listener) {
		mPreferences.registerOnSharedPreferenceChangeListener(listener);
	}

	/**
	 */
	@Override
	public void unregisterOnSharedPreferenceChangeListener(@NonNull final OnSharedPreferenceChangeListener listener) {
		mPreferences.unregisterOnSharedPreferenceChangeListener(listener);
	}

	/**
	 */
	@Override
	public Map<String, ?> getAll() {
		return mPreferences.getAll();
	}

	/**
	 */
	@Override
	public boolean contains(@NonNull final String key) {
		return mPreferences.contains(key);
	}

	/**
	 */
	@Nullable
	@Override
	public String getString(@NonNull final String key, @Nullable final String defValue) {
		return mPreferences.getString(key, defValue);
	}

	/**
	 */
	@Nullable
	@Override
	public Set<String> getStringSet(@NonNull final String key, @Nullable final Set<String> defValues) {
		return mPreferences.getStringSet(key, defValues);
	}

	/**
	 */
	@Override
	public int getInt(@NonNull final String key, final int defValue) {
		return mPreferences.getInt(key, defValue);
	}

	/**
	 */
	@Override
	public long getLong(@NonNull final String key, final long defValue) {
		return mPreferences.getLong(key, defValue);
	}

	/**
	 */
	@Override
	public float getFloat(@NonNull final String key, final float defValue) {
		return mPreferences.getFloat(key, defValue);
	}

	/**
	 */
	@Override
	public boolean getBoolean(@NonNull final String key, final boolean defValue) {
		return mPreferences.getBoolean(key, defValue);
	}

	/**
	 */
	@NonNull
	@Override
	public Editor edit() {
		return mPreferences.edit();
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
