/*
 * =================================================================================================
 *                       Copyright (C) 2011 The Android Open Source Project
 * =================================================================================================
 *                             Copyright (C) Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.pager.adapter;

import android.annotation.SuppressLint;
import android.app.Fragment;
import android.app.FragmentManager;
import android.app.FragmentTransaction;
import android.os.Build;
import android.os.Bundle;
import android.os.Parcelable;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.view.PagerAdapter;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;

import java.util.ArrayList;
import java.util.List;

/**
 * A {@link PagerAdapter} implementation that represents each page as a {@link Fragment}. This pager
 * adapter also handles saving and restoring state of its fragment.
 * <p>
 * This version of the pager adapter is more useful when there are a large number of pages, working
 * more like a list view. When pages are not visible to a user, their entire fragment may be destroyed,
 * only keeping the saved state of that fragment. This allows to the pager to hold on to much less
 * memory associated with each visited page as compared to {@link FragmentPagerAdapter} at the cost
 * of potentially more overhead when switching between pages.
 * <p>
 * <b>Note</b>, that when using FragmentPagerAdapter, the host ViewPager must have a valid ID set,
 * otherwise this adapter implementation will throw an exception.
 * <p>
 * The inheritance hierarchies only need to implement {@link #getItem(int)} and
 * {@link #getCount()} to become a fully working adapters.
 */
@SuppressLint("LongLogTag")
public abstract class FragmentStatePagerAdapter extends PagerAdapter {

	/**
	 * Interface ===================================================================================
	 */

	/**
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	private static final String TAG = "FragmentStatePagerAdapter";

	/**
	 * Bundle key used to store saved states within this adapter's saved state.
	 *
	 * @see #saveState()
	 * @see #restoreState(Parcelable, ClassLoader)
	 */
	private static final String BUNDLE_SAVED_STATES = TAG + ".BUNDLE.SavedStates";

	/**
	 * Static members ==============================================================================
	 */

	/**
	 * Members =====================================================================================
	 */

	/**
	 * Fragment manager used to manage fragments provided by this pager adapter.
	 */
	private final FragmentManager mFragmentManager;

	/**
	 * Boolean flag indicating whether this pager adapter should create tags for its items or not.
	 *
	 * @see #makeItemTag(int, long)
	 */
	private boolean mMakeItemTags = false;

	/**
	 * Current pending fragment transaction to be committed in {@link #finishUpdate(ViewGroup)}.
	 */
	private FragmentTransaction mPendingTransaction = null;

	/**
	 * List used to store saved state for by this adapter instantiated fragments.
	 */
	private final List<Fragment.SavedState> mSavedState = new ArrayList<>(10);

	/**
	 * List used to store by this adapter instantiated fragments.
	 */
	private final List<Fragment> mFragments = new ArrayList<>(10);

	/**
	 * Fragment that is currently the primary item in the associated pager.
	 */
	Fragment mPrimaryItem = null;

	/**
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new instance of FragmentStatePagerAdapter with the specified <var>fragmentManager</var>.
	 *
	 * @param fragmentManager The manager used to manage fragments provided by the pager adapter.
	 */
	public FragmentStatePagerAdapter(@NonNull FragmentManager fragmentManager) {
		this.mFragmentManager = fragmentManager;
	}

	/**
	 * Methods =====================================================================================
	 */

	/**
	 */
	@Override
	public void startUpdate(ViewGroup container) {
		if (container.getId() == View.NO_ID) throw new IllegalStateException(
				"ViewPager with adapter " + this + " requires a view id!"
		);
	}

	/**
	 */
	@Override
	@SuppressLint("CommitTransaction")
	public Object instantiateItem(ViewGroup container, int position) {
		// If we already have this item instantiated, there is nothing to do. This can happen when
		// we are restoring the entire pager from its saved state, where the fragment manager has
		// already taken care of restoring the fragments we previously had instantiated.
		if (mFragments.size() > position) {
			final Fragment fragment = mFragments.get(position);
			if (fragment != null) return fragment;
		}
		if (mPendingTransaction == null) {
			// Will be committed in call to finishUpdate(...).
			this.mPendingTransaction = mFragmentManager.beginTransaction();
		}
		final long itemId = getItemId(position);
		final Fragment fragment = getItem(position);
		if (mSavedState.size() > position) {
			final Fragment.SavedState fragmentSavedState = mSavedState.get(position);
			if (fragmentSavedState != null) {
				fragment.setInitialSavedState(fragmentSavedState);
			}
		}
		while (mFragments.size() <= position) {
			mFragments.add(null);
		}
		fragment.setMenuVisibility(false);
		FragmentPagerAdapter.setUserVisibleHint(fragment, false);
		mFragments.set(position, fragment);
		if (PagerAdaptersConfig.DEBUG_LOG_ENABLED) {
			Log.v(TAG, "Adding item(id: " + itemId + ", fragment: " + fragment + ").");
		}
		final int containerId = container.getId();
		mPendingTransaction.add(containerId, fragment, makeItemTag(containerId, itemId));
		return fragment;
	}

	/**
	 * Sets a boolean flag indicating whether this pager adapter should make tags for its items or
	 * not.
	 * <p>
	 * This feature is by default <b>disabled</b>.
	 *
	 * @param makeItemTags {@code True} to enable making of item tags, {@code false} otherwise.
	 * @see #makeItemTag(int, long)
	 */
	protected final void setMakeItemTags(boolean makeItemTags) {
		this.mMakeItemTags = makeItemTags;
	}

	/**
	 * Makes a tag for an item with the given <var>itemId</var>.
	 * <p>
	 * By default making of item tags is <b>disabled</b> and the default implementation makes tag
	 * in the following format:
	 * <pre>
	 * "android:pager:CONTAINER_ID:ITEM_ID"
	 * </pre>
	 * This feature may be enabled/disabled via {@link #setMakeItemTags(boolean)}.
	 *
	 * @param containerId Id of the associated view pager container.
	 * @param itemId      Id of the item for which to create its corresponding tag.
	 * @return Item's tag or {@code null} if this adapter does not create tags for its items.
	 */
	@Nullable
	protected String makeItemTag(@IdRes int containerId, long itemId) {
		return mMakeItemTags ? "android:pager:" + containerId + ":" + itemId : null;
	}

	/**
	 * Returns a unique identifier for the item at the specified <var>position</var>.
	 * <p>
	 * The default implementation returns the given position. The inheritance hierarchies should
	 * override this method if the positions of items can change.
	 *
	 * @param position Position from the range of size of this this adapter.
	 * @return Unique identifier for the item at the requested position.
	 */
	public long getItemId(int position) {
		return position;
	}

	/**
	 * Instantiates a new {@link Fragment} associated with the specified <var>position</var>.
	 *
	 * @see #instantiateItem(ViewGroup, int)
	 */
	@NonNull
	public abstract Fragment getItem(int position);

	/**
	 */
	@Override
	public void setPrimaryItem(ViewGroup container, int position, Object object) {
		final Fragment fragment = (Fragment) object;
		if (mPrimaryItem != fragment) {
			if (mPrimaryItem != null) {
				this.mPrimaryItem.setMenuVisibility(false);
				FragmentPagerAdapter.setUserVisibleHint(mPrimaryItem, false);
			}
			if (fragment != null) {
				fragment.setMenuVisibility(true);
				FragmentPagerAdapter.setUserVisibleHint(fragment, true);
			}
			this.mPrimaryItem = fragment;
		}
	}

	/**
	 */
	@Override
	public boolean isViewFromObject(View view, Object object) {
		return ((Fragment) object).getView() == view;
	}

	/**
	 */
	@Override
	@SuppressLint("CommitTransaction")
	public void destroyItem(ViewGroup container, int position, Object object) {
		if (mPendingTransaction == null) {
			// Will be committed in call to finishUpdate(...).
			this.mPendingTransaction = mFragmentManager.beginTransaction();
		}
		final Fragment fragment = (Fragment) object;
		if (PagerAdaptersConfig.DEBUG_LOG_ENABLED) {
			Log.v(TAG, "Removing item(id: " + getItemId(position) + ", fragment: " + fragment + ", view: " + fragment.getView() + ").");
		}
		while (mSavedState.size() <= position) {
			this.mSavedState.add(null);
		}
		this.mSavedState.set(
				position,
				fragment.isAdded() ? mFragmentManager.saveFragmentInstanceState(fragment) : null
		);
		this.mFragments.set(position, null);
		this.mPendingTransaction.remove(fragment);
	}

	/**
	 */
	@Override
	public void finishUpdate(ViewGroup container) {
		if (mPendingTransaction != null) {
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
				this.mPendingTransaction.commitNowAllowingStateLoss();
			} else {
				this.mPendingTransaction.commitAllowingStateLoss();
			}
			this.mPendingTransaction = null;
		}
	}

	/**
	 */
	@Override
	public Parcelable saveState() {
		Bundle state = null;
		if (mSavedState.size() > 0) {
			state = new Bundle();
			final Fragment.SavedState[] savedStates = new Fragment.SavedState[mSavedState.size()];
			mSavedState.toArray(savedStates);
			state.putParcelableArray(BUNDLE_SAVED_STATES, savedStates);
		}
		for (int i = 0; i < mFragments.size(); i++) {
			final Fragment fragment = mFragments.get(i);
			if (fragment != null && fragment.isAdded()) {
				if (state == null) {
					state = new Bundle();
				}
				mFragmentManager.putFragment(state, "f" + i, fragment);
			}
		}
		return state;
	}

	/**
	 */
	@Override
	public void restoreState(Parcelable state, ClassLoader loader) {
		if (state != null) {
			Bundle bundle = (Bundle) state;
			bundle.setClassLoader(loader);
			final Parcelable[] savedStates = bundle.getParcelableArray(BUNDLE_SAVED_STATES);
			mSavedState.clear();
			mFragments.clear();
			if (savedStates != null && savedStates.length > 0) {
				for (final Parcelable savedState : savedStates) {
					mSavedState.add((Fragment.SavedState) savedState);
				}
			}
			final Iterable<String> keys = bundle.keySet();
			for (String key : keys) {
				if (key.startsWith("f")) {
					int index = Integer.parseInt(key.substring(1));
					final Fragment fragment = mFragmentManager.getFragment(bundle, key);
					if (fragment != null) {
						while (mFragments.size() <= index) {
							mFragments.add(null);
						}
						fragment.setMenuVisibility(false);
						mFragments.set(index, fragment);
					} else {
						Log.w(TAG, "Bad fragment at key " + key);
					}
				}
			}
		}
	}

	/**
	 * Inner classes ===============================================================================
	 */
}