/*
 * =================================================================================================
 *                       Copyright (C) 2011 The Android Open Source Project
 * =================================================================================================
 *                             Copyright (C) 2016 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.pager.adapter;

import android.annotation.SuppressLint;
import android.app.Fragment;
import android.app.FragmentManager;
import android.app.FragmentTransaction;
import android.os.Build;
import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.view.PagerAdapter;
import android.util.Log;
import android.view.View;
import android.view.ViewGroup;

/**
 * A {@link PagerAdapter} implementation that represents each page as a {@link Fragment} that is
 * persistently kept in the fragment manager as long as a user can return to the page.
 * <p>
 * This version of the pager adapter is best for use when there are a handful of typically more static
 * fragments to be paged through, such as a set of tabs. The fragment of each page the user visits
 * will be kept in memory, though its view hierarchy may be destroyed when not visible. This can
 * result in using a significant amount of memory since fragment instances can hold on to an arbitrary
 * amount of state. For larger sets of pages, consider {@link FragmentStatePagerAdapter}.
 * <p>
 * <b>Note</b>, that when using FragmentPagerAdapter, the host ViewPager must have a valid ID set,
 * otherwise this adapter implementation will throw an exception.
 * <p>
 * The inheritance hierarchies only need to implement {@link #getItem(int)} and
 * {@link #getCount()} to become a fully working adapters.
 */
public abstract class FragmentPagerAdapter extends PagerAdapter {

	/**
	 * Interface ===================================================================================
	 */

	/**
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	private static final String TAG = "FragmentPagerAdapter";

	/**
	 * Static members ==============================================================================
	 */

	/**
	 * Members =====================================================================================
	 */

	/**
	 * Fragment manager used to manage fragments provided by this pager adapter.
	 */
	private final FragmentManager mFragmentManager;

	/**
	 * Boolean flag indicating whether this pager adapter should create tags for its items or not.
	 *
	 * @see #makeItemTag(int, long)
	 */
	private boolean mMakeItemTags = true;

	/**
	 * Current pending fragment transaction to be committed in {@link #finishUpdate(ViewGroup)}.
	 */
	private FragmentTransaction mPendingTransaction = null;

	/**
	 * Fragment that is currently the primary item in the associated pager.
	 */
	Fragment mPrimaryItem = null;

	/**
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new instance of FragmentPagerAdapter with the specified <var>fragmentManager</var>.
	 *
	 * @param fragmentManager The manager used to manage fragments provided by the pager adapter.
	 */
	public FragmentPagerAdapter(@NonNull FragmentManager fragmentManager) {
		this.mFragmentManager = fragmentManager;
	}

	/**
	 * Methods =====================================================================================
	 */

	/**
	 * Sets a user visible hint for the given <var>fragment</var>.
	 *
	 * @param fragment Fragment for which to set user visible hint flag.
	 * @param visible {@code True} to set user visible hint, {@code false} otherwise.
	 * @see Fragment#setUserVisibleHint(boolean)
	 */
	static void setUserVisibleHint(@NonNull Fragment fragment, boolean visible) {
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH_MR1) {
			fragment.setUserVisibleHint(visible);
		}
	}

	/**
	 */
	@Override
	public void startUpdate(ViewGroup container) {
		if (container.getId() == View.NO_ID) throw new IllegalStateException(
				"ViewPager with adapter " + this + " requires a view id!"
		);
	}

	/**
	 */
	@Override
	@SuppressLint("CommitTransaction")
	public Object instantiateItem(ViewGroup container, int position) {
		if (mPendingTransaction == null) {
			// Will be committed in call to finishUpdate(...).
			this.mPendingTransaction = mFragmentManager.beginTransaction();
		}
		final long itemId = getItemId(position);
		// Check whether we already have this fragment instantiated.
		final String tag = makeItemTag(container.getId(), itemId);
		Fragment fragment = mFragmentManager.findFragmentByTag(tag);
		if (fragment != null) {
			if (PagerAdaptersConfig.DEBUG_LOG_ENABLED) {
				Log.v(TAG, "Attaching item(id: " + itemId + ", fragment: " + fragment + ").");
			}
			mPendingTransaction.attach(fragment);
		} else {
			fragment = getItem(position);
			if (PagerAdaptersConfig.DEBUG_LOG_ENABLED) {
				Log.v(TAG, "Adding item(id: " + itemId + ", fragment: " + fragment + ").");
			}
			final int containerId = container.getId();
			mPendingTransaction.add(containerId, fragment, makeItemTag(containerId, itemId));
		}
		if (fragment != mPrimaryItem) {
			fragment.setMenuVisibility(false);
			setUserVisibleHint(fragment, false);
		}
		return fragment;
	}

	/**
	 * Sets a boolean flag indicating whether this pager adapter should make tags for its items or
	 * not.
	 * <p>
	 * This feature is by default <b>enabled</b>.
	 *
	 * @param makeItemTags {@code True} to enable making of item tags, {@code false} otherwise.
	 * @see #makeItemTag(int, long)
	 */
	protected final void setMakeItemTags(boolean makeItemTags) {
		this.mMakeItemTags = makeItemTags;
	}

	/**
	 * Makes a tag for an item with the given <var>itemId</var>.
	 * <p>
	 * By default making of item tags is <b>enabled</b> and the default implementation makes tag
	 * in the following format:
	 * <pre>
	 * "android:pager:CONTAINER_ID:ITEM_ID"
	 * </pre>
	 * This feature may be enabled/disabled via {@link #setMakeItemTags(boolean)}.
	 *
	 * @param containerId Id of the associated view pager container.
	 * @param itemId      Id of the item for which to create its corresponding tag.
	 * @return Item's tag or {@code null} if this adapter does not create tags for its items.
	 */
	@Nullable
	protected String makeItemTag(@IdRes int containerId, long itemId) {
		return mMakeItemTags ? "android:pager:" + containerId + ":" + itemId : null;
	}

	/**
	 * Returns a unique identifier for the item at the specified <var>position</var>.
	 * <p>
	 * The default implementation returns the given position. The inheritance hierarchies should
	 * override this method if the positions of items can change.
	 *
	 * @param position Position from the range of size of this this adapter.
	 * @return Unique identifier for the item at the requested position.
	 */
	public long getItemId(int position) {
		return position;
	}

	/**
	 * Instantiates a new {@link Fragment} associated with the specified <var>position</var>.
	 *
	 * @see #instantiateItem(ViewGroup, int)
	 */
	@NonNull
	public abstract Fragment getItem(int position);

	/**
	 */
	@Override
	public void setPrimaryItem(ViewGroup container, int position, Object object) {
		final Fragment fragment = (Fragment) object;
		if (mPrimaryItem != fragment) {
			if (mPrimaryItem != null) {
				this.mPrimaryItem.setMenuVisibility(false);
				setUserVisibleHint(mPrimaryItem, false);
			}
			if (fragment != null) {
				fragment.setMenuVisibility(true);
				setUserVisibleHint(fragment, true);
			}
			this.mPrimaryItem = fragment;
		}
	}

	/**
	 */
	@Override
	public boolean isViewFromObject(View view, Object object) {
		return ((Fragment) object).getView() == view;
	}

	/**
	 */
	@Override
	@SuppressLint("CommitTransaction")
	public void destroyItem(ViewGroup container, int position, Object object) {
		if (mPendingTransaction == null) {
			// Will be committed in call to finishUpdate(...).
			this.mPendingTransaction = mFragmentManager.beginTransaction();
		}
		if (PagerAdaptersConfig.DEBUG_LOG_ENABLED) {
			Log.v(TAG, "Detaching item(id: " + getItemId(position) + ", fragment: " + object + ", view: " + ((Fragment) object).getView() + ").");
		}
		mPendingTransaction.detach((Fragment) object);
	}

	/**
	 */
	@Override
	public void finishUpdate(ViewGroup container) {
		if (mPendingTransaction != null) {
			if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
				this.mPendingTransaction.commitNowAllowingStateLoss();
			} else {
				this.mPendingTransaction.commitAllowingStateLoss();
			}
			this.mPendingTransaction = null;
		}
	}

	/**
	 * Inner classes ===============================================================================
	 */
}