/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.form;

import android.support.annotation.IdRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StringRes;
import android.text.Editable;
import android.text.SpannableStringBuilder;
import android.text.TextUtils;
import android.view.View;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import universum.studios.android.ui.widget.EditLayout;

/**
 * A {@link BaseFormComponent} implementation which wraps form {@link EditLayout} into component that
 * can be validated by {@link FormValidator} when validating a form in which is such a layout presented.
 *
 * @author Martin Albedinsky
 */
public class FormEditableLayout extends BaseFormComponent<EditLayout, CharSequence> {

	/**
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "FormEditableLayout";

	/**
	 * Regular expression to check for <b>not</b> empty text.
	 */
	public static final String NOT_EMPTY_REG_EXP = "^(.+)$";

	/**
	 * Interface ===================================================================================
	 */

	/**
	 * Static members ==============================================================================
	 */

	/**
	 * Members =====================================================================================
	 */

	/**
	 * Matcher used to check input of this form component's view.
	 */
	private Matcher mEditMatcher;

	/**
	 * Resource id of the regular expression for {@link #mEditMatcher}.
	 */
	private final int mRegExpResId;

	/**
	 * Constructors ================================================================================
	 */

	/**
	 * Same as {@link #FormEditableLayout(int, String)} with {@link FormEditable#NOT_EMPTY_REG_EXP}
	 * for <var>regExp</var> parameter.
	 */
	public FormEditableLayout(@IdRes int id) {
		this(id, NOT_EMPTY_REG_EXP);
	}

	/**
	 * Creates a new instance of FormEditableLayout component for an editable input layout with the
	 * specified <var>id</var> and the specified regular expression.
	 *
	 * @param id     Id used to find the editable layout within the form view hierarchy of
	 *               the associated {@link FormValidator} via {@link FormValidator#findViewById(int)}.
	 * @param regExp The desired regular expression that is used to validate editable
	 *               value obtained from the editable view during the validation process.
	 */
	public FormEditableLayout(@IdRes int id, @NonNull String regExp) {
		super(id, null);
		this.mEditMatcher = Pattern.compile(regExp).matcher("");
		this.mRegExpResId = -1;
	}

	/**
	 * Creates a new instance of FormEditableLayout component for an an editable input layout with
	 * the specified <var>id</var>.
	 *
	 * @param id          Id used to find the editable layout within the form view hierarchy of
	 *                    the associated {@link FormValidator} via {@link FormValidator#findViewById(int)}.
	 * @param regExpResId Resource id of the desired regular expression that is used to validate editable
	 *                    value obtained from the editable view during the validation process.
	 */
	public FormEditableLayout(@IdRes int id, @StringRes int regExpResId) {
		super(id, null);
		this.mRegExpResId = regExpResId;
	}

	/**
	 * Same as {@link #FormEditableLayout(EditLayout, String)} with {@link FormEditable#NOT_EMPTY_REG_EXP}
	 * for <var>regExp</var> parameter.
	 */
	public FormEditableLayout(@NonNull EditLayout editable) {
		this(editable, NOT_EMPTY_REG_EXP);
	}

	/**
	 * Same as {@link #FormEditableLayout(EditLayout, String)} for regular expression resource id.
	 *
	 * @param regExpResId Resource id of the desired regular expression.
	 */
	public FormEditableLayout(@NonNull EditLayout editable, @StringRes int regExpResId) {
		this(editable, editable.getResources().getString(regExpResId));
	}

	/**
	 * Creates a new instance of FormEditable component for the given <var>editable</var> view with
	 * the specified regular expression.
	 *
	 * @param editable Editable input layout, which should be validated.
	 * @param regExp   The desired regular expression used to validate editable value obtained from
	 *                 the <var>editable</var> view during the validation process.
	 * @see BaseFormComponent#BaseFormComponent(int, View)
	 */
	public FormEditableLayout(@NonNull EditLayout editable, @NonNull String regExp) {
		super(editable.getId(), editable);
		this.mEditMatcher = Pattern.compile(regExp).matcher("");
		this.mRegExpResId = -1;
	}

	/**
	 * Methods =====================================================================================
	 */

	/**
	 */
	@Override
	public boolean hasInputChanged() {
		return mLastValidatedInput == null || !TextUtils.equals(mLastValidatedInput, getInput());
	}

	/**
	 */
	@Override
	protected boolean onValidate(@NonNull CharSequence input) {
		return mEditMatcher.reset(input).matches();
	}

	/**
	 */
	@NonNull
	@Override
	public CharSequence getInput() {
		final Editable editable = getEditableInput();
		return editable == null ? "" : new SpannableStringBuilder(editable);
	}

	/**
	 * Returns the current editable value of the attached text field view.
	 *
	 * @return Current editable, or {@code null} if view of this component is {@code null}.
	 */
	@Nullable
	public Editable getEditableInput() {
		return mView == null ? null : mView.getEditableText();
	}

	/**
	 */
	@Override
	public void setError(@NonNull CharSequence errorMessage) {
		setHasError(true);
		if (mView != null) mView.setError(errorMessage);
	}

	/**
	 */
	@Override
	public void clearError() {
		super.clearError();
		if (mView != null) mView.clearError();
	}

	/**
	 */
	@Override
	protected void onAttachedToValidator(@NonNull FormValidator validator) {
		super.onAttachedToValidator(validator);
		if (mRegExpResId > 0) {
			this.mEditMatcher = Pattern.compile(obtainString(mRegExpResId)).matcher("");
		}
	}

	/**
	 * Inner classes ===============================================================================
	 */
}
