/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.font.widget;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build;
import android.support.annotation.AttrRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StyleRes;
import android.util.AttributeSet;
import android.view.accessibility.AccessibilityEvent;
import android.view.accessibility.AccessibilityNodeInfo;
import android.widget.TextView;

import universum.studios.android.font.Font;
import universum.studios.android.font.FontApplier;
import universum.studios.android.font.FontApplierWidget;

/**
 * A {@link TextView} and {@link FontApplierWidget} implementation that supports applying of custom
 * font via {@link #setFont(Font)}.
 * <p>
 * This font widget implementation also allows to specify custom font via
 * {@link universum.studios.android.font.R.attr#uiFont uiFont} attribute through Xml {@link AttributeSet}
 * or through text appearance style which may be set via {@link #setTextAppearance(int)}.
 *
 * @author Martin Albedinsky
 */
public class FontTextView extends TextView implements FontApplierWidget {

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "FontTextView";

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Applier to which is this font widget delegating applying of {@link Font} whenever one of
	 * {@link #setFont(String)}, {@link #setFont(Font)} or {@link #setTextAppearance(Context, int)}
	 * methods is invoked.
	 */
	private FontApplier mFontApplier = FontApplier.DEFAULT;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet)} without attributes.
	 */
	public FontTextView(@NonNull final Context context) {
		this(context, null);
	}

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet, int)} with {@link android.R.attr#textViewStyle}
	 * as attribute for default style.
	 */
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs) {
		this(context, attrs, android.R.attr.textViewStyle);
	}

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet, int, int)} with {@code 0} as default style.
	 */
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs, @AttrRes final int defStyleAttr) {
		super(context, attrs, defStyleAttr);
		if (!isInEditMode()) this.mFontApplier.applyFont(this, attrs, defStyleAttr, 0);
	}

	/**
	 * Creates a new instance of FontTextView for the given <var>context</var>.
	 *
	 * @param context      Context in which will be the new view presented.
	 * @param attrs        Set of Xml attributes used to configure the new instance of view.
	 * @param defStyleAttr Attribute which contains a reference to a default style resource for view
	 *                     within a theme of the given context.
	 * @param defStyleRes  Resource id of the default style for the view.
	 */
	@SuppressWarnings("unused")
	@TargetApi(Build.VERSION_CODES.LOLLIPOP)
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs, @AttrRes final int defStyleAttr, @StyleRes final int defStyleRes) {
		super(context, attrs, defStyleAttr, defStyleRes);
		if (!isInEditMode()) this.mFontApplier.applyFont(this, attrs, defStyleAttr, defStyleRes);
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 */
	@Override
	@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
	public void onInitializeAccessibilityEvent(@NonNull final AccessibilityEvent event) {
		super.onInitializeAccessibilityEvent(event);
		event.setClassName(FontTextView.class.getName());
	}

	/**
	 */
	@Override
	@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
	public void onInitializeAccessibilityNodeInfo(@NonNull final AccessibilityNodeInfo info) {
		super.onInitializeAccessibilityNodeInfo(info);
		info.setClassName(FontTextView.class.getName());
	}

	/**
	 */
	@Override
	public void setFontApplier(@NonNull final FontApplier applier) {
		this.mFontApplier = applier;
	}

	/**
	 */
	@Override
	public void setFont(@NonNull final String fontPath) {
		this.mFontApplier.applyFont(this, fontPath);
	}

	/**
	 */
	@Override
	public void setFont(@NonNull final Font font) {
		this.mFontApplier.applyFont(this, font);
	}

	/**
	 */
	@Override
	@SuppressWarnings("deprecation")
	public void setTextAppearance(@NonNull final Context context, @StyleRes final int resId) {
		super.setTextAppearance(context, resId);
		this.mFontApplier.applyFont(this, resId);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
