/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License 
 * you may obtain at
 * 
 * 		http://www.apache.org/licenses/LICENSE-2.0
 * 
 * You can redistribute, modify or publish any part of the code written within this file but as it 
 * is described in the License, the software distributed under the License is distributed on an 
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.font.widget;

import android.annotation.TargetApi;
import android.content.Context;
import android.os.Build;
import android.support.annotation.AttrRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StyleRes;
import android.util.AttributeSet;
import android.view.accessibility.AccessibilityEvent;
import android.view.accessibility.AccessibilityNodeInfo;
import android.widget.TextView;

import universum.studios.android.font.Font;
import universum.studios.android.font.FontWidget;
import universum.studios.android.font.util.FontApplier;

/**
 * A {@link TextView} and {@link FontWidget} implementation that supports setting of custom font via
 * {@link #setFont(Font)}.
 * <p>
 * This font widget implementation also allows to specify custom font via
 * {@link universum.studios.android.font.R.attr#uiFont uiFont} attribute through Xml {@link AttributeSet}
 * or through text appearance style which may be set via {@link #setTextAppearance(int)}.
 *
 * @author Martin Albedinsky
 */
public class FontTextView extends TextView implements FontWidget {

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	// private static final String TAG = "FontTextView";

	/*
	 * Static members ==============================================================================
	 */

	/*
	 * Members =====================================================================================
	 */

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet)} without attributes.
	 */
	public FontTextView(@NonNull final Context context) {
		this(context, null);
	}

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet, int)} with {@link android.R.attr#textViewStyle}
	 * as attribute for default style.
	 */
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs) {
		this(context, attrs, android.R.attr.textViewStyle);
	}

	/**
	 * Same as {@link #FontTextView(Context, AttributeSet, int, int)} with {@code 0} as default style.
	 */
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs, @AttrRes final int defStyleAttr) {
		super(context, attrs, defStyleAttr);
		this.init(context, attrs, defStyleAttr, 0);
	}

	/**
	 * Creates a new instance of FontTextView for the given <var>context</var>.
	 *
	 * @param context      Context in which will be the new view presented.
	 * @param attrs        Set of Xml attributes used to configure the new instance of view.
	 * @param defStyleAttr Attribute which contains a reference to a default style resource for view
	 *                     within a theme of the given context.
	 * @param defStyleRes  Resource id of the default style for the view.
	 */
	@SuppressWarnings("unused")
	@TargetApi(Build.VERSION_CODES.LOLLIPOP)
	public FontTextView(@NonNull final Context context, @Nullable final AttributeSet attrs, @AttrRes final int defStyleAttr, @StyleRes final int defStyleRes) {
		super(context, attrs, defStyleAttr, defStyleRes);
		this.init(context, attrs, defStyleAttr, defStyleRes);
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Called from one of constructors of this view to perform its initialization.
	 * <p>
	 * Initialization is done via parsing of the specified <var>attrs</var> set and obtaining for
	 * this view specific data from it that can be used to configure this new view instance. The
	 * specified <var>defStyleAttr</var> and <var>defStyleRes</var> are used to obtain default data
	 * from the current theme provided by the specified <var>context</var>.
	 */
	private void init(final Context context, final AttributeSet attrs, final int defStyleAttr, final int defStyleRes) {
		if (!isInEditMode()) FontApplier.applyFont(this, attrs, defStyleAttr, defStyleRes);
	}

	/**
	 */
	@Override
	@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
	public void onInitializeAccessibilityEvent(@NonNull final AccessibilityEvent event) {
		super.onInitializeAccessibilityEvent(event);
		event.setClassName(FontTextView.class.getName());
	}

	/**
	 */
	@Override
	@TargetApi(Build.VERSION_CODES.ICE_CREAM_SANDWICH)
	public void onInitializeAccessibilityNodeInfo(@NonNull final AccessibilityNodeInfo info) {
		super.onInitializeAccessibilityNodeInfo(info);
		info.setClassName(FontTextView.class.getName());
	}

	/**
	 */
	@Override
	public void setFont(@NonNull final String fontPath) {
		FontApplier.applyFont(this, fontPath);
	}

	/**
	 */
	@Override
	public void setFont(@NonNull final Font font) {
		FontApplier.applyFont(this, font);
	}

	/**
	 */
	@Override
	@SuppressWarnings("deprecation")
	public void setTextAppearance(@NonNull final Context context, @StyleRes final int resId) {
		super.setTextAppearance(context, resId);
		FontApplier.applyFont(this, resId);
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
