/*
 * =================================================================================================
 *                             Copyright (C) 2017 Universum Studios
 * =================================================================================================
 *         Licensed under the Apache License, Version 2.0 or later (further "License" only).
 * -------------------------------------------------------------------------------------------------
 * You may use this file only in compliance with the License. More details and copy of this License
 * you may obtain at
 *
 * 		http://www.apache.org/licenses/LICENSE-2.0
 *
 * You can redistribute, modify or publish any part of the code written within this file but as it
 * is described in the License, the software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES or CONDITIONS OF ANY KIND.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * =================================================================================================
 */
package universum.studios.android.font;

import android.content.Context;
import android.content.res.AssetManager;
import android.content.res.Resources;
import android.content.res.TypedArray;
import android.graphics.Typeface;
import android.support.annotation.AttrRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.annotation.StyleRes;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.Log;

import java.io.File;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Font is a simple wrapper class around graphics {@link Typeface} object. Each Font instance is
 * required to be created with a relative path to its corresponding .ttf font file that is placed
 * within <b>assets</b> directory. A new Font instance may be created directly via {@link #Font(String)}
 * constructor, however it is encouraged to create new instances of Font via one of {@link #create(String)},
 * {@link #create(Context, int)} or {@link #create(Context, AttributeSet, int, int)} factory methods,
 * because these methods ensure that already created fonts are reused across entire Android application
 * which definitely improves performance as creation of a single Typeface object via
 * {@link Typeface#createFromAsset(AssetManager, String)} is a costly operation.
 * <p>
 * This library supports only <b>.ttf</b> font files. Each font file is required to be placed within
 * assets directory of the corresponding application. If {@link FontConfig#STRICT_STORAGE} is enabled,
 * then font files must be placed in <b>assets/font/</b> directory, but may be grouped in sub-directories.
 * Relative path to a specific <b>true typeface font</b> file may or may not have specified <b>.ttf</b>
 * suffix. If the suffix is not specified, this class ads such suffix automatically.
 * <p>
 * Typeface object of a specific Font instance may be obtained via {@link #getTypeface(Context)}.
 *
 * @author Martin Albedinsky
 */
public final class Font {

	/*
	 * Constants ===================================================================================
	 */

	/**
	 * Log TAG.
	 */
	private static final String TAG = "Font";

	/**
	 * Suffix for original <b>true Typeface</b> fonts.
	 * <p>
	 * Constant value: <b>.ttf</b>
	 */
	public static final String TTF_SUFFIX = ".ttf";

	/**
	 * Sub-directory within an application assets folder, where should be placed all custom font files
	 * if strict mode is enabled via {@link FontConfig#STRICT_STORAGE}.
	 * <p>
	 * This folder however may contain custom sub-directories to group related fonts together.
	 * <p>
	 * Constant value: <b>font/</b>
	 */
	public static final String FONT_DIR = "font" + File.separator;

	/*
	 * Interface ===================================================================================
	 */

	/*
	 * Static members ==============================================================================
	 */

	/**
	 * Matcher for <b>.ttf</b> file suffix.
	 */
	private static final Matcher TTF_SUFFIX_MATCHER = Pattern.compile("(.*)\\.ttf").matcher("");

	/**
	 * Cache containing already created fonts. The fonts are stored and mapped by theirs corresponding
	 * font path.
	 */
	private static final Map<String, Font> sCache = new HashMap<>(5);

	/*
	 * Members =====================================================================================
	 */

	/**
	 * Path to the .ttf file within an application assets folder which is represented by this font
	 * instance.
	 */
	private final String mFontPath;

	/**
	 * Typeface created by this font from the corresponding .ttf file.
	 */
	private Typeface mTypeFace;

	/*
	 * Constructors ================================================================================
	 */

	/**
	 * Creates a new instance of Font with the specified path to font file within assets folder.
	 * <p>
	 * <b>Note, that empty path or {@code null} are not valid arguments.</b>
	 * <p>
	 * <b>Only .ttf font files are supported!</b>
	 *
	 * @param fontPath Relative path to the .ttf file (with or without .ttf suffix) placed within
	 *                 an application assets directory which will be represented by the newly created
	 *                 Font instance.
	 * @throws IllegalArgumentException If the given <var>fontPath</var> is empty.
	 */
	public Font(@NonNull final String fontPath) {
		if (TextUtils.isEmpty(fontPath)) {
			throw new IllegalArgumentException("Font path cannot be empty.");
		}
		this.mFontPath = fontPath;
	}

	/*
	 * Methods =====================================================================================
	 */

	/**
	 * Creates a new instance of Font with the font path obtained from the given <var>attrs</var> set.
	 *
	 * @param context      Context used to process the given attributes set.
	 * @param attrs        Attributes set which that is expected to have {@link R.attr#uiFont uiFont}
	 *                     attribute specified with font path.
	 * @param defStyleAttr An attribute of the default style presented within the current theme which
	 *                     supplies default attributes for {@link TypedArray}.
	 * @param defStyleRes  Resource id of the default style used when the <var>defStyleAttr</var> is 0
	 *                     or the current theme does not have style for <var>defStyleAttr</var> specified.
	 * @return New or cached instance of Font or {@code null} if font path parsed from the given attributes
	 * set is invalid.
	 * @see #create(String)
	 * @see #create(Context, int)
	 */
	@Nullable
	public static Font create(@NonNull final Context context, @NonNull final AttributeSet attrs, @AttrRes final int defStyleAttr, @StyleRes final int defStyleRes) {
		final Resources.Theme theme = context.getTheme();
		final TypedArray typedArray = theme.obtainStyledAttributes(attrs, new int[]{R.attr.uiFont}, defStyleAttr, defStyleRes);
		final String fontPath = typedArray.getString(0);
		typedArray.recycle();
		return TextUtils.isEmpty(fontPath) ? null : create(fontPath);
	}

	/**
	 * Creates a new instance of Font with the font path obtained from the given <var>style</var>.
	 *
	 * @param context Context used to process the given style.
	 * @param style   Resource id of the desired style that is expected to have {@link R.attr#uiFont uiFont}
	 *                attribute specified with font path.
	 * @return New or cached instance of Font or {@code null} if font path parsed from the given style
	 * is invalid.
	 * @see #create(String)
	 * @see #create(Context, AttributeSet, int, int)
	 */
	@Nullable
	public static Font create(@NonNull final Context context, @StyleRes final int style) {
		final Resources.Theme theme = context.getTheme();
		final TypedArray typedArray = theme.obtainStyledAttributes(style, new int[]{R.attr.uiFont});
		final String fontPath = typedArray.getString(0);
		typedArray.recycle();
		return TextUtils.isEmpty(fontPath) ? null : create(fontPath);
	}

	/**
	 * Creates an instance of Font with the given font path.
	 *
	 * @return New or cached instance of Font for the specified path.
	 * @throws IllegalArgumentException If the given <var>fontPath</var> is empty.
	 * @see #create(Context, AttributeSet, int, int)
	 * @see #create(Context, int)
	 * @see #Font(String)
	 */
	@NonNull
	public static Font create(@NonNull final String fontPath) {
		if (TextUtils.isEmpty(fontPath)) {
			throw new IllegalArgumentException("Font path cannot be empty.");
		}
		final String fullFontPath = buildFullFontPath(fontPath);
		if (sCache.containsKey(fullFontPath)) {
			if (FontConfig.DEBUG_LOG_ENABLED) {
				Log.v(TAG, "Re-using cached font for path(" + fullFontPath + ").");
			}
			return sCache.get(fullFontPath);
		}
		if (FontConfig.DEBUG_LOG_ENABLED) {
			Log.v(TAG, "Creating and caching new font for path(" + fullFontPath + ").");
		}
		final Font font = new Font(fullFontPath);
		sCache.put(fullFontPath, font);
		return font;
	}

	/**
	 * Builds a full path to the .ttf font file from the given <var>fontPath</var>.
	 *
	 * @param fontPath The font path that should be build as full path.
	 * @return Full path containing the specified font path with added {@link #TTF_SUFFIX} if not
	 * presented in the specified path and also prepended with {@link #FONT_DIR} if {@link FontConfig#STRICT_STORAGE}
	 * is enabled.
	 */
	private static String buildFullFontPath(final String fontPath) {
		final String fontFolder = FontConfig.STRICT_STORAGE ? FONT_DIR : "";
		return fontFolder + (TTF_SUFFIX_MATCHER.reset(fontPath).matches() ? fontPath : fontPath + TTF_SUFFIX);
	}

	/**
	 * Returns the path to the font file of this font instance.
	 *
	 * @return Full path to the .ttf file within an application assets folder.
	 * @see #Font(String)
	 */
	@NonNull
	public String getFontPath() {
		return mFontPath;
	}

	/**
	 * Returns instance of the Typeface which is created for the .ttf file of this font instance.
	 * <p>
	 * Subsequent calls to this method will return already created and cached Typeface object.
	 *
	 * @param context Valid context used to create requested TypeFace.
	 * @return Typeface associated with this font instance.
	 * @throws IllegalArgumentException If font file at the path specified for this font does not
	 *                                  exist or is not a valid .ttf file.
	 * @see Typeface#createFromAsset(android.content.res.AssetManager, String)
	 */
	@NonNull
	public Typeface getTypeface(@NonNull final Context context) {
		if (mTypeFace == null) {
			try {
				this.mTypeFace = Typeface.createFromAsset(context.getAssets(), mFontPath);
			} catch (Exception e) {
				throw new IllegalArgumentException(
						"Cannot create Typeface for font path(" + mFontPath + "). " +
								"Font file at the specified path does not exist or is not a valid .ttf file.",
						e
				);
			}
		}
		return mTypeFace;
	}

	/*
	 * Inner classes ===============================================================================
	 */
}
