/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 * 
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.util

import androidx.annotation.NonNull

/**
 * A convenience factory class which may be used to create [Error] instances.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
object Errors {

    /**
     * Creates a new Error with the specified [code] and [description].
     *
     * The create error will have specified [NONE][Cause.NONE] cause.
     *
     * @param code The desired code for the new error.
     * @param description The desired description for the new error. Default is [EMPTY][Description.EMPTY].
     * @return Error ready to be propagated further.
     */
    @JvmStatic @JvmOverloads
    fun create(
        @NonNull code: String,
        @NonNull description: Description = Description.EMPTY
    ) = create(code, description, Cause.NONE)

    /**
     * Creates a new Error with the specified [code] and [cause].
     *
     * The create error will have specified [EMPTY][Description.EMPTY] description.
     *
     * @param code The desired code for the new error.
     * @param cause The cause due to which the new error occurred.
     * @return Error ready to be propagated further.
     */
    @JvmStatic fun create(
        @NonNull code: String,
        @NonNull cause: Throwable
    ) = create(code, Description.EMPTY, cause)

    /**
     * Creates a new Error with the specified [code], [description] and [cause].
     *
     * @param code The desired code for the new error.
     * @param description The desired description for the new error.
     * @param cause The cause due to which the new error occurred.
     * @return Error ready to be propagated further.
     */
    @JvmStatic fun create(
        @NonNull code: String,
        @NonNull description: Description,
        @NonNull cause: Throwable
    ): Error = ErrorException.create(code, description, cause)
}