/*
 * *************************************************************************************************
 *                                 Copyright 2018 Universum Studios
 * *************************************************************************************************
 *                  Licensed under the Apache License, Version 2.0 (the "License")
 * -------------------------------------------------------------------------------------------------
 * You may not use this file except in compliance with the License. You may obtain a copy of the
 * License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the License
 * is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express
 * or implied.
 *
 * See the License for the specific language governing permissions and limitations under the License.
 * *************************************************************************************************
 */
package universum.studios.android.arkhitekton.util

import androidx.annotation.NonNull
import androidx.annotation.Nullable

/**
 * Convenient implementation of [Throwable] which may be used as base class for concrete causes.
 *
 * This class also provides [NONE][Cause.NONE] and [UNKNOWN][Cause.UNKNOWN] cause instances.
 *
 * @author Martin Albedinsky
 * @since 1.0
 */
open class Cause : Throwable {

    /**
     * Contract for [Cause] element.
     */
    companion object Contract {

        /**
         * Instance of [Throwable] which may be used as empty object to indicate no particular cause.
         */
        @NonNull @JvmField
        val NONE: Throwable = object : Cause("", null) {

            /*
             */
            @NonNull override fun toString() = "NONE"
        }

        /**
         * Instance of [Throwable] which may be used to indicate an 'unknown' cause.
         */
        @NonNull @JvmField
        val UNKNOWN: Throwable = object : Cause("", NONE) {

            /*
             */
            @NonNull override fun toString() = "UNKNOWN"
        }
    }

    /**
     * Creates a new instance of empty Cause with empty message and [NONE][Cause.NONE] as additional cause.
     */
    constructor() : this("", NONE)

    /**
     * Creates a new instance of Cause with the specified [message] and [NONE][Cause.NONE] as additional cause.
     *
     * @param message The desired detailed message describing the new cause.
     */
    constructor(@NonNull message: String) : this(message, NONE)

    /**
     * Creates a new instance of Cause with the specified [cause].
     *
     * @param cause The additional cause of the new cause.
     */
    constructor(@NonNull cause: Throwable) : this(cause.toString(), cause)

    /**
     * Creates a new instance of Cause with the specified [message] and [cause].
     *
     * @param message The desired detailed message describing the new cause. May be `null`.
     * @param cause The additional cause of the new cause. May be `null`.
     */
    constructor(@Nullable message: String?, @Nullable cause: Throwable?) : super(message, cause)
}